/*
 *    rttyrx.c  --  RTTY demodulator
 *
 *    Copyright (C) 2001, 2002, 2003
 *      Tomi Manninen (oh2bns@sral.fi)
 *
 *    This file is part of gMFSK.
 *
 *    gMFSK is free software; you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation; either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    gMFSK is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with gMFSK; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

#include <stdio.h>

#include "../main/trx.h"
#include "rtty.h"
#include "../misc/misc.h"
#include "baudot.h"
#include "rttypar.h"
#include <math.h>

enum {LOWPASS,BANDPASS,HIGHPASS};

#define FS 8000

/**********************************************************
IIR Biquad Filters.
**********************************************************/

extern float biq_lp(float x,float *pcoef,float *buf)
{
	float y;
	y =(*pcoef++)*(x+buf[0]+buf[0]+buf[1]);
	y+=(*pcoef++)*buf[2];
	y+=(*pcoef)*buf[3];

	buf[1]=buf[0]; buf[0]=x;
	buf[3]=buf[2]; buf[2]=y;

	return y;
}

extern float biq_bp(float x,float *pcoef,float *buf)
{
	float y;
	y =(*pcoef++)*(x-buf[1]);
	y+=(*pcoef++)*buf[2];
	y+=(*pcoef)*buf[3];

	buf[1]=buf[0]; buf[0]=x;
	buf[3]=buf[2]; buf[2]=y;

	return y;
}

extern float biq_hp(float x,float *pcoef,float *buf)
{
	float y;
	y =(*pcoef++)*(x-buf[0]-buf[0]+buf[1]);
	y+=(*pcoef++)*buf[2];
	y+=(*pcoef)*buf[3];

	buf[1]=buf[0]; buf[0]=x;
	buf[3]=buf[2]; buf[2]=y;

	return y;
}

/*******************************************************
Coef. Calculation
*******************************************************/
void gen_coef(int tipo, float f0, float Q, float *pcoef)
{
	float w0,a,d;

	w0=(float)(2.*M_PI*f0);

	//Prewharping
	w0=(float)(2.0*FS*tan(w0/FS/2.0));

	a=FS/w0;
	d=(float)4.*a*a +2.*a/Q +1.;

	switch(tipo){
	case LOWPASS:  (*pcoef++)=(float)1.0/d;	break;
	case BANDPASS: (*pcoef++)=(float)2.*a/Q/d;	break;
	case HIGHPASS: (*pcoef++)=(float)4.*a*a/d;	break;
	}

	(*pcoef++)=(float)(8.*a*a -2.)/d;
	(*pcoef++)=-(float)(4.*a*a -2.*a/Q +1.)/d;
}

void rtty_set_filters(struct rtty *s, float f0, float f1, float dr)
{
	gen_coef(BANDPASS,f0,(float)f0/dr/2.0,s->bp0_c);	/* Space filter */
	gen_coef(BANDPASS,f1,(float)f1/dr/2.0,s->bp1_c);	/* Mark filter  */
	gen_coef(LOWPASS,dr,0.5412f,s->lp1_c);	/* Low-Pass order-4 Butt. filter */
	gen_coef(LOWPASS,dr,1.3066f,s->lp2_c);

	//for (i=0;i<4;i++) lp1_b[i]=lp2_b[i]=bp0_b[i]=bp1_b[i]=0.0;
}

/*******************************************************
FSK & CW demodulation
*******************************************************/
float demodulator(struct rtty *s, float sample)
{
	float xs,xm,y;

	xs=biq_bp(biq_bp(sample,s->bp0_c,s->bp0_b),s->bp0_c,s->bp0_b1);
		//xs=biq_bp(sample,bp0_c,bp0_b);
	xm=biq_bp(biq_bp(sample,s->bp1_c,s->bp1_b),s->bp1_c,s->bp1_b1);
		//xm=biq_bp(sample,bp1_c,bp1_b);
	xs*=xs;		/* xs RMS */
	xm*=xm;		/* xm RMS */
	y=biq_lp(biq_lp(xm-xs,s->lp1_c,s->lp1_b),s->lp2_c,s->lp2_b);
	return y;
}

static void update_syncscope(struct rtty *s)
{
	float data[MaxSymLen];
	int i, j;

	for (i = 0; i < s->symbollen; i++) {
		j = (i + s->pipeptr) % s->symbollen;
		data[i] = (float)0.5 + 0.85 * s->pipe[j] / s->shift;
	}

	trx_set_scope(data, s->symbollen, FALSE);
}

static unsigned char bitreverse(unsigned char in, int n)
{
	unsigned char out = 0;
	int i;

	for (i = 0; i < n; i++)
		out = (out << 1) | ((in >> i) & 1);

	return out;
}

static int decode_char(struct rtty *s)
{
	unsigned int parbit, par, data;

	parbit = (s->rxdata >> s->nbits) & 1;
	par = rtty_parity(s->rxdata, s->nbits, s->parity);

	if (s->parity != RTTY_PARITY_NONE && parbit != par) {
//		fprintf(stderr, "P");
		return 0;
	}

	data = s->rxdata & ((1 << s->nbits) - 1);

	if (s->msb)
		data = bitreverse(data, s->nbits);

	if (s->nbits == 5)
		return baudot_dec(&s->rxmode, data);

	return data;
}

static int rttyrx(Trx *trx, int bit)
{
	struct rtty *s = (struct rtty *) trx->modem;
	int flag = 0;
	unsigned char c;

	switch (s->rxstate) {
	case RTTY_RX_STATE_IDLE:
		if (!bit) {
			s->rxstate = RTTY_RX_STATE_START;
			s->counter = s->symbollen / 2;
		}
		break;

	case RTTY_RX_STATE_START:
		if (--s->counter == 0) {
			if (!bit) {
				s->rxstate = RTTY_RX_STATE_DATA;
				s->counter = s->symbollen;
				s->bitcntr = 0;
				s->rxdata = 0;
				flag = 1;
			} else
				s->rxstate = RTTY_RX_STATE_IDLE;
		}
		break;

	case RTTY_RX_STATE_DATA:
		if (--s->counter == 0) {
			s->rxdata |= bit << s->bitcntr++;
			s->counter = s->symbollen;
			flag = 1;
		}

		if (s->bitcntr == s->nbits) {
			if (s->parity == RTTY_PARITY_NONE)
				s->rxstate = RTTY_RX_STATE_STOP;
			else
				s->rxstate = RTTY_RX_STATE_PARITY;
		}
		break;

	case RTTY_RX_STATE_PARITY:
		if (--s->counter == 0) {
			s->rxstate = RTTY_RX_STATE_STOP;
			s->rxdata |= bit << s->bitcntr++;
			s->counter = s->symbollen;
			flag = 1;
		}
		break;

	case RTTY_RX_STATE_STOP:
		if (--s->counter == 0) {
			if (bit) {
				c = decode_char(s);
				trx_put_rx_char(trx, c);
				flag = 1;
			} else {
//				fprintf(stderr, "F");
			}
			s->rxstate = RTTY_RX_STATE_STOP2;
			s->counter = s->symbollen / 2;
		}
		break;

	case RTTY_RX_STATE_STOP2:
		if (--s->counter == 0)
			s->rxstate = RTTY_RX_STATE_IDLE;
		break;
	}

	return flag;
}

int rtty_rxprocess(Trx *trx, short *buf, int len)
{
	struct rtty *s = (struct rtty *) trx->modem;
	int bit, rev;

	rev = (trx->reverse != 0) ^ (s->reverse != 0);

	rtty_set_filters(s, 
		(trx->frequency - (s->shift >> 1)) >> FREQ_FRAC_BITS, 
		(trx->frequency + (s->shift >> 1)) >> FREQ_FRAC_BITS, 
		trx->rtty_baud);

	while (len-- > 0) {
		float f = demodulator(s, *buf++);
		if (rev)
			bit = (f > 0.0);
		else
			bit = (f < 0.0);

		if (rttyrx(trx, bit) && trx->afcon) {
			if (f > 0.0)
				f = f - s->shift / 2;
			else
				f = f + s->shift / 2;

//				fprintf(stderr, "bit=%d f=% f\n", bit, f);

//FIXME
//			if (fabs(f) < s->shift / 2)
//				trx_set_freq(trx->frequency + f / 256);
		}
	}

	return 0;
}
