/*
 *    psk31tx.c  --  PSK31 modulator
 *
 *    Copyright (C) 2001, 2002, 2003, 2004
 *      Tomi Manninen (oh2bns@sral.fi)
 *
 *    This file is part of gMFSK.
 *
 *    gMFSK is free software; you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation; either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    gMFSK is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with gMFSK; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

#include <ctype.h>

#include "psk31.h"
#include "varicode.h"
#include "../misc/mixer.h"
#include "../main/sndmodem.h"

static void send_symbol_raw(Trx *trx, int re1, int im1, int re2, int im2)
{
	struct psk31 *s			  = (struct psk31 *) trx->modem;
	unsigned	  iPhaseDelta = (unsigned)((65536.0 * 65536.0) * trx->frequency / (SampleRate << FREQ_FRAC_BITS));
	int			  i;

	for (i = 0; i < s->symbollen; i++) {
		if (1) {
			int ival = (s->txshape[i] * re1 + (32767 - s->txshape[i]) * re2);
			int qval = (s->txshape[i] * im1 + (32767 - s->txshape[i]) * im2);
			trx->outbuf[i] = mixer_mix2(ival, qval, s->phaseacc) >> 16;
		} else {
			int ival = (re1 == re2) ? (32767 * re1) : 
				(s->txshape[i] * re1 + s->txshape2[i] * re2);
	//			(s->txshape[i] * re1 + (32767 - s->txshape[i]) * re2);
			int qval = (im1 == im2) ? (32767 * im1) : 
				(s->txshape[i] * im1 + s->txshape2[i] * im2);
	//			(s->txshape[i] * im1 + (32767 - s->txshape[i]) * im2);
			trx->outbuf[i] = mixer_mix2(ival, qval, s->phaseacc) >> 16;
		}
		s->phaseacc += iPhaseDelta;
	}

	sndmodem_write(trx->pModem, trx->outbuf, s->symbollen);
}

static void send_symbol(Trx *trx, int sym)
{
	struct psk31 *s = (struct psk31 *) trx->modem;
	complex z, symbol;

	if (s->qpsk && trx->reverse)
		sym = (4 - sym) & 3;

	/* differential QPSK modulation - top bit flipped */
	switch (sym) {
	case 0:
		c_re(z) = -1;	/* 180 degrees */
		c_im(z) =  0;
		break;
	case 1:
		c_re(z) =  0;	/* 270 degrees */
		c_im(z) = -1;
		break;
	case 2:
		c_re(z) =  1;	/*   0 degrees */
		c_im(z) =  0;
		break;
	case 3:
		c_re(z) =  0;	/*  90 degrees */
		c_im(z) =  1;
		break;
	}

	c_re(symbol) = c_re(s->prevsymbol) * c_re(z) - c_im(s->prevsymbol) * c_im(z);
	c_im(symbol) = c_re(s->prevsymbol) * c_im(z) + c_im(s->prevsymbol) * c_re(z);

	send_symbol_raw(trx, s->prevsymbol.re, s->prevsymbol.im, symbol.re, symbol.im);

	/* save the current symbol */
	s->prevsymbol = symbol;
}

static void send_bit(Trx *trx, int bit)
{
	struct psk31 *s = (struct psk31 *) trx->modem;
	unsigned int sym;

	if (s->qpsk)
		sym = encoder_encode(s->enc, bit);
	else
		sym = bit << 1;

	send_symbol(trx, sym);
}

static void send_char(Trx *trx, unsigned char c)
{
	char *code;

	code = psk_varicode_encode(c);

	while (*code) {
		send_bit(trx, (*code - '0'));
		code++;
	}

	send_bit(trx, 0);
	send_bit(trx, 0);
}

static void flush_tx(Trx *trx)
{
	struct psk31 *s = (struct psk31 *) trx->modem;
	int i;

	/* flush the encoder (QPSK only) */
	if (s->qpsk) {
		send_bit(trx, 0);
		send_bit(trx, 0);
		send_bit(trx, 0);
		send_bit(trx, 0);
		send_bit(trx, 0);
	}

	/* DCD off sequence (unmodulated carrier) */
	for (i = 0; i < 31; i++)
		send_symbol(trx, 2);
	send_symbol_raw(trx, s->prevsymbol.re, s->prevsymbol.im, 0, 0);
	sndmodem_tx_flush(trx->pModem);
}

int psk31_txprocess(Trx *trx)
{
	struct psk31 *s = (struct psk31 *) trx->modem;
	int c;

	if (trx->tune) {
		trx->tune = 0;
		send_symbol(trx, 0);	/* send phase reversals */
		return 0;
	}

	if (s->preamble > 0) {
		s->preamble --;
		if (s->preamble == 31)
			send_symbol_raw(trx, 0, 0, 1, 0);
		else
			send_symbol(trx, 0);	/* send phase reversals */
		return 0;
	}

	c = trx_get_tx_char(trx);

	/* TX buffer empty */
	if (c == -1) {
		/* stop if requested to... */
		if (trx->stopflag) {
			flush_tx(trx);
			return -1;	/* we're done */
		}

		send_symbol(trx, 0);	/* send phase reversals */
		send_symbol(trx, 0);
		return 0;
	}

	send_char(trx, c);
	trx_put_echo_char(trx, c);

	return 0;
}
