/*
 *    psk31rx.c  --  PSK31 demodulator
 *
 *    Copyright (C) 2001, 2002, 2003
 *      Tomi Manninen (oh2bns@sral.fi)
 *
 *    This file is part of gMFSK.
 *
 *    gMFSK is free software; you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation; either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    gMFSK is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with gMFSK; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

#include <stdio.h>

#ifdef _WIN32_WCE
#define _ASSERT(x) 
#else
	#include <crtdbg.h>
#endif /* _WIN32_WCE */

#include "psk31.h"
#include "../misc/fixedpoint.h"
#include "../misc/filter.h"
#include "varicode.h"
#include "coeff.h"


#include "../misc/mixer.h"

static void rx_bit(Trx *trx, int bit)
{
	struct psk31 *s = (struct psk31 *) trx->modem;
	int c;

	s->shreg = (s->shreg << 1) | !!bit;

	if ((s->shreg & 3) == 0) {
		c = psk_varicode_decode(s->shreg >> 2);

		if (c != -1)
			trx_put_rx_char(trx, c);

		s->shreg = 0;
	}
}

static void rx_qpsk(Trx *trx, int bits)
{
	struct psk31 *s = (struct psk31 *) trx->modem;
	unsigned char sym[2];
	int c;

	if (s->qpsk && trx->reverse)
		bits = (4 - bits) & 3;

	sym[0] = (bits & 1) ? 255 : 0;
	sym[1] = (bits & 2) ? 0 : 255;		/* top bit is flipped */

	c = viterbi_decode(s->dec, sym, NULL);

	if (c != -1) {
		rx_bit(trx, c & 0x80);
		rx_bit(trx, c & 0x40);
		rx_bit(trx, c & 0x20);
		rx_bit(trx, c & 0x10);
		rx_bit(trx, c & 0x08);
		rx_bit(trx, c & 0x04);
		rx_bit(trx, c & 0x02);
		rx_bit(trx, c & 0x01);
	}
}

static void rx_symbol(Trx *trx, complex symbol)
{
	struct psk31 *s = (struct psk31 *) trx->modem;
	int bits, n;
	int diffphasex = s->prevsymbol.re * symbol.re + s->prevsymbol.im * symbol.im;
	int diffphasey = s->prevsymbol.re * symbol.im - s->prevsymbol.im * symbol.re;
	int phase      = iatan2(diffphasey, diffphasex);

//	_ASSERT(phase >= -32768 && phase < 32768);
	if (phase < 0)
		phase += 1 << 16; // + 2 * PI
//	_ASSERT(phase >= 0 && phase < 65536);

	if (s->qpsk) {
		phase += 1 << 13;
		phase &= 0xFFFF;
		bits = (phase >> 14) & 3; // (phase / M_PI_2 + 0.5) & 3;
		n = 4;
	} else {
		phase += 1 << 14;
		phase &= 0xFFFF;
		bits = ((phase >> 15) & 1) << 1; // (((int) (phase / M_PI + 0.5)) & 1) << 1;
		n = 2;
	}

	if (trx->squelchon) {
		unsigned int sqlphase = phase << ((n == 2) ? 17 : 18);
		int		     ampl;
		s->quality.re -= s->quality.re >> 6;						// *= 63/64;
		s->quality.re += mixer_osc_16(sqlphase + 0x40000000) >> 6;	// += cos(sqlphase) / 64
		s->quality.im -= s->quality.im >> 6;						// *= 63/64;
		s->quality.im += mixer_osc_16(sqlphase) >> 6;				// += sin(sqlphase) / 64
		ampl		   = s->quality.re * s->quality.re + s->quality.im * s->quality.im;
		_ASSERT(fabs(s->quality.re) < 32768);
		_ASSERT(fabs(s->quality.im) < 32768);

		s->dcdshreg = (s->dcdshreg << 2) | bits;
		switch (s->dcdshreg) {
		case 0xAAAAAAAA:	/* DCD on by preamble */
			s->dcd = TRUE;
			c_re(s->quality) = 32767;
			c_im(s->quality) = 0;
			break;

		case 0:			/* DCD off by postamble */
			s->dcd = FALSE;
			c_re(s->quality) = 0;
			c_im(s->quality) = 0;
			break;

		default:
		{
			int threshold = 107374 * trx->iSquelchThreshold * trx->iSquelchThreshold;
//			int threshold = 4294 * trx->iSquelchThreshold * trx->iSquelchThreshold * trx->iSquelchThreshold;
			_ASSERT(fabs(s->quality.re) < 32678);
			_ASSERT(fabs(s->quality.im) < 32678);
			s->dcd = ampl > threshold;
/*
			{
				char bufout[256];
				sprintf(bufout, "metric: %03f%%, %ld, %ld\n", 100.0 * ampl / threshold, ampl, threshold);
				OutputDebugString(bufout);
			}
*/
			break;
		}
		}
	}

	trx_set_phase(phase, s->dcd);

	if (s->dcd || ! trx->squelchon) {
		if (s->qpsk)
			rx_qpsk(trx, bits);
		else
			rx_bit(trx, !bits);
		if (trx->afcon) {
			int error = phase << (s->qpsk ? 2 : 1);
			error &= 0xFFFF;
			if (error > 32767)
				error -= 0xFFFF;
//			_ASSERT(error >= - 0x8000 && error < 0x8000);
			error  *= 8000;
			error /= s->symbollen;
			error >>= 16 + 2 + 4 - FREQ_FRAC_BITS;  // div 2 * M_PI
			trx->frequency -= error;
		}
	}

	s->prevsymbol = symbol;
}

static void update_syncscope(Trx *trx)
{
	struct psk31 *s = (struct psk31 *) trx->modem;
	int    data[16];
	unsigned int i, ptr;

	ptr = s->pipeptr - 24;

	for (i = 0; i < 16; ++ i)
		data[i] = s->pipe[ptr++ % PipeLen];

	trx_set_scope(data, 16, TRUE);
}

short aSyncCoeffs[16] = {
	-12785,
	-12294,
	-11330,
	-9931,
	-8150,
	-6056,
	-3729,
	-1259,
	1259,
	3729,
	6056,
	8150,
	9931,
	11330,
	12294,
};

int psk31_rxprocess(Trx *trx, short *buf, int len)
{
	struct psk31 *s = (struct psk31 *) trx->modem;
	complex z;
	int i;

	while (len -- > 0) {
		// bits of trx->frequency = 12 /* 0-4kHz */ + FREQ_FRAC_BITS
//		unsigned iPhaseDelta2 = (unsigned)((65536.0 * 65536.0) * trx->frequency / (SampleRate << FREQ_FRAC_BITS));
//		unsigned iPhaseDelta = 
//			(((trx->frequency << (32 - (12 + FREQ_FRAC_BITS))) / SampleRate) << 12) + (1 << 11);
		unsigned iPhaseDelta = 33554 /* 2^32  / (SampleRate << FREQ_FRAC_BITS) = 3354.432 */ * trx->frequency;

		/* Mix with the internal NCO */
		mixer_mix(*(buf ++), s->phaseacc, &z);
		s->phaseacc += iPhaseDelta;
		z.re >>= 16;
		z.im >>= 16;

		/* Filter and downsample by 16 or 8 */
		if (filter_run(s->fir1, z, &z)) {
			int  sum = 0;
			int  idx;
			UINT ampl;
			int  level  = 0;
			int  ftemp  = 0;
			int  ampsum = 0;

			/* Do the second filtering */
			z.re >>= 16;
			z.im >>= 16;
			filter_run(s->fir2, z, &z);
			z.re >>= 16;
			z.im >>= 16;

			/* save amplitude value for the sync scope */
			ampl  = (UINT)(z.re * z.re) + (UINT)(z.im * z.im);
			level = isqrt(ampl);
			s->pipe[s->pipeptr] = level;

//			trx_set_scope(s->pipe, PipeLen, TRUE);

			/* Now the sync correction routine... */
			idx = s->bitclk >> 20;
//			idx = (int) s->bitclk;
			s->syncbuf[idx] = level;

			for (i = 0; i < 16; ++ i) {
				ftemp  += aSyncCoeffs[i] * s->syncbuf[i];
				ampsum += s->syncbuf[i];
			}
			ftemp = (ampsum != 0) ? (ftemp / ampsum) : 0;
			for (i = 0; i < 8; i++)
				sum += s->syncbuf[i];
			for (i = 8; i < 16; i++)
				sum -= s->syncbuf[i];
			sum = (ampsum != 0) ? ((sum << 12) / ampsum) : 0;
//			s->bitclk += ftemp << 5; // or 6
//			s->bitclk += ftemp << 4;
//			s->bitclk -= sum << (24 - 12 - 5);
// Works well for noisy signals
//			s->bitclk -= sum << (24 - 12 - 7);
			s->bitclk -= sum << (24 - 12 - 6);
/*
			{
				char bufout[256];
				sprintf(bufout, "syncdif: %ld\n", syncdif);
				OutputDebugString(bufout);
			}
*/

//FIXME test the sound card RX clock correction
			s->bitclk += 0x100000;
//			s->bitclk += 0x1028F5; // + 1%
//			s->bitclk += 0x0FD772; // - 1%
			if (s->bitclk & 0xFF000000) {
				// here we are at the center of the bit
				s->bitclk &= 0xFFFFFF;
				rx_symbol(trx, z);
				update_syncscope(trx);
			}

			s->pipeptr = (s->pipeptr + 1) % PipeLen;
		}
	}

	return 0;
}
