/*
 *    psk31.c  --  PSK31 modem
 *
 *    Copyright (C) 2001, 2002, 2003
 *      Tomi Manninen (oh2bns@sral.fi)
 *
 *    This file is part of gMFSK.
 *
 *    gMFSK is free software; you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation; either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    gMFSK is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with gMFSK; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

#include <stdlib.h>
#include <stdio.h>

#include "psk31.h"
#include "../misc/fixedpoint.h"
#include "../misc/filter.h"
#include "../misc/mixer.h"
#include "coeff.h"

#define	K	5
#define	POLY1	0x17
#define	POLY2	0x19

static void psk31_txinit(Trx *trx)
{
	struct psk31 *s = (struct psk31 *) trx->modem;

	s->phaseacc = 0;

	c_re(s->prevsymbol) = 1;
	c_im(s->prevsymbol) = 0;

	s->preamble = 32;

	s->shreg = 0;
}

static void psk31_rxinit(Trx *trx)
{
	struct psk31 *s = (struct psk31 *) trx->modem;

	s->phaseacc = 0;

	c_re(s->prevsymbol) = MAXINT;
	c_im(s->prevsymbol) = 0;

	c_re(s->quality) = 0;
	c_im(s->quality) = 0;

	s->shreg = 0;
	s->dcdshreg = 0;
	s->dcd = 0;

	s->bitclk = 0;
}

static void psk31_free(struct psk31 *s)
{
	if (s) {
		filter_free(s->fir1);
		filter_free(s->fir2);

		encoder_free(s->enc);
		viterbi_free(s->dec);

		free(s->txshape);
		free(s->txshape2);

		free(s);
	}
}

static void psk31_destructor(Trx *trx)
{
	struct psk31 *s = (struct psk31 *) trx->modem;

	psk31_free(s);

	trx->modem = NULL;
	trx->txinit = NULL;
	trx->rxinit = NULL;
	trx->txprocess = NULL;
	trx->rxprocess = NULL;
	trx->destructor = NULL;
}

void psk31_init(Trx *trx)
{
	struct psk31 *s;
	int i;

	if ((s = calloc(1, sizeof(struct psk31))) == NULL)
		return;

	switch (trx->mode) {
	case MODE_BPSK31:
		s->symbollen = 256;
		s->qpsk = 0;
		break;
	case MODE_QPSK31:
		s->symbollen = 256;
		s->qpsk = 1;
		break;
	case MODE_PSK63:
		s->symbollen = 128;
		s->qpsk = 0;
		break;
	default:
		psk31_free(s);
		return;
	}

	s->fir1 = filter_init(PSK31_FIRLEN, s->symbollen / 16, psk31_fir1c, psk31_fir1c);
	s->fir2 = filter_init(PSK31_FIRLEN, 1, psk31_fir2c, psk31_fir2c);

	if (!s->fir1 || !s->fir2) {
		psk31_free(s);
		return;
	}

	if (s->qpsk) {
		s->enc = encoder_init(K, POLY1, POLY2);
		s->dec = viterbi_init(K, POLY1, POLY2);

		if (!s->enc || !s->dec) {
			psk31_free(s);
			return;
		}
	}

	if ((s->txshape = malloc(s->symbollen * sizeof(double))) == NULL || 
		(s->txshape2 = malloc(s->symbollen * sizeof(double))) == NULL) {
		psk31_free(s);
		return;
	}

	/* raised cosine shape for the transmitter */
	if (1) {
		for (i = 0; i < s->symbollen; i++)
			s->txshape[i] = (int)((0.5 * cos(i * M_PI / s->symbollen) + 0.5) * 32767);
	} else {
//		double tau = 22.0 /* ohm */ * 0.0001 /* F */; // elecraft
		double tau = 22.0 /* ohm */ * 0.0001 /* F */;
		for (i = 0; i < (s->symbollen >> 1); ++ i) {
			s->txshape [i]		    			 = (int)(exp(- (double)i / (tau * (double)SampleRate)) * 32767);
			s->txshape2[i]						 = 0;
			s->txshape [(s->symbollen >> 1) + i] = 0;
			s->txshape2[(s->symbollen >> 1) + i] = 32767 - s->txshape[i];
		}
	}

    trx->modem = s;

	trx->txinit = psk31_txinit;
	trx->rxinit = psk31_rxinit;

	trx->txprocess = psk31_txprocess;
	trx->rxprocess = psk31_rxprocess;

	trx->destructor = psk31_destructor;

//	trx->samplerate = SampleRate;
//	trx->bandwidth = (double) SampleRate / s->symbollen;
}
