
#ifdef _WIN32_WCE
#define _ASSERT(bool_val) 
#else /* _WIN32_WCE */
#include <assert.h>
#include <crtdbg.h>
#endif /* _WIN32_WCE */

#include <math.h>
#include "int_fft.h"

/*      fix_fft.c - Fixed-point Fast Fourier Transform  */
/*
        fix_fft()       perform FFT or inverse FFT
        window()        applies a Hanning window to the (time) input
        fix_loud()      calculates the loudness of the signal, for
                        each freq point. Result is an integer array,
                        units are dB (values will be negative).
        iscale()        scale an integer value by (numer/denom).
        FIX_MPY()       perform fixed-point multiplication.
        Sinewave[1024]  sinewave normalized to 32767 (= 1.0).
        Loudampl[100]   Amplitudes for lopudnesses from 0 to -99 dB.
        Low_pass        Low-pass filter, cutoff at sample_freq / 4.


        All data are fixed-point short integers, in which
        -32768 to +32768 represent -1.0 to +1.0. Integer arithmetic
        is used for speed, instead of the more natural floating-point.

        For the forward FFT (time -> freq), fixed scaling is
        performed to prevent arithmetic overflow, and to map a 0dB
        sine/cosine wave (i.e. amplitude = 32767) to two -6dB freq
        coefficients; the one in the lower half is reported as 0dB
        by fix_loud(). The return value is always 0.

        For the inverse FFT (freq -> time), fixed scaling cannot be
        done, as two 0dB coefficients would sum to a peak amplitude of
        64K, overflowing the 32k range of the fixed-point integers.
        Thus, the fix_fft() routine performs variable scaling, and
        returns a value which is the number of bits LEFT by which
        the output must be shifted to get the actual amplitude
        (i.e. if fix_fft() returns 3, each value of fr[] and fi[]
        must be multiplied by 8 (2**3) for proper scaling.
        Clearly, this cannot be done within the fixed-point short
        integers. In practice, if the result is to be used as a
        filter, the scale_shift can usually be ignored, as the
        result will be approximately correctly normalized as is.


        TURBO C, any memory model; uses inline assembly for speed
        and for carefully-scaled arithmetic.

        Written by:  Tom Roberts  11/8/89
        Made portable:  Malcolm Slaney 12/15/94 malcolm@interval.com

                Timing on a Macintosh PowerBook 180.... (using Symantec C6.0)
                        fix_fft (1024 points)             8 ticks
                        fft (1024 points - Using SANE)  112 Ticks
                        fft (1024 points - Using FPU)    11

*/

#define FIX_MPY(A,B)    ((short)(((long)(A) * (long)(B))>>15))

#define N_WAVE          (8 * 1024)  /* dimension of Sinewave[] */
#define LOG2_N_WAVE     13          /* log2(N_WAVE) */
#define N_LOUD          100         /* dimension of Loudampl[] */
#ifndef fixed
#define fixed short
#endif

fixed *Sinewave = NULL;
extern fixed Loudampl[N_LOUD];  /* placed at end of this file for clarity */
int db_from_ampl(fixed re, fixed im);

BOOL fix_fft_initialize()
{
	int    i;
	double dValueMax = 32767.0;
	fixed  iValueMax = 32767;

	Sinewave = (fixed*)malloc(N_WAVE * sizeof(fixed));
	if (Sinewave == 0)
		return FALSE;

	for (i = 0; i < (N_WAVE >> 1); ++ i) {
		double dValue = dValueMax * sin(6.283185307179586476925286766559 * ((double)i + 0.5) / (double)N_WAVE);
		int    iValue = (int)(dValue + 0.5);
		if (iValue < 0)
			iValue = iValueMax;
		Sinewave[i] = iValue;
	}
	for (; i < N_WAVE; ++ i)
		Sinewave[i] = - Sinewave[i - (N_WAVE >> 1)];

	return TRUE;
}

void fix_fft_destroy()
{
	free(Sinewave);
	Sinewave = NULL;
}

/*
    fix_fft() - perform fast Fourier transform.

    if n>0 FFT is done, if n<0 inverse FFT is done
    fr[n],fi[n] are real,imaginary arrays, INPUT AND RESULT.
    size of data = 2**m
    set inverse to 0=dft, 1=idft
*/
int fix_fft(fixed fr[], fixed fi[], int m, int inverse)
{
	int mr,nn,i,j,l,k,istep, n, scale, shift;
	fixed qr,qi,tr,ti,wr,wi;

	n = 1<<m;

	if(n > N_WAVE)
		return -1;

	mr = 0;
	nn = n - 1;
	scale = 0;

	/* decimation in time - re-order data */
	for(m=1; m<=nn; ++m) {
		l = n;
		do {
			l >>= 1;
		} while(mr+l > nn);
		mr = (mr & (l-1)) + l;

		if(mr <= m) 
			continue;
		tr = fr[m];
		fr[m] = fr[mr];
		fr[mr] = tr;
		ti = fi[m];
		fi[m] = fi[mr];
		fi[mr] = ti;
	}

    l = 1;
    k = LOG2_N_WAVE-1;
    while(l < n) {
		if(inverse) {
			/* variable scaling, depending upon data */
			shift = 0;
			for(i=0; i<n; ++i) {
				j = fr[i];
				if(j < 0)
					j = -j;
				m = fi[i];
				if(m < 0)
					m = -m;
				if(j > 16383 || m > 16383) {
					shift = 1;
					break;
				}
			}
			if(shift)
				++scale;
		} else {
			/* fixed scaling, for proper normalization -
				there will be log2(n) passes, so this
				results in an overall factor of 1/n,
				distributed to maximize arithmetic accuracy. */
			shift = 1;
		}
		/* it may not be obvious, but the shift will be performed
			on each data point exactly once, during this pass. */
		istep = l << 1;
		for(m=0; m<l; ++m) {
			j = m << k;
			/* 0 <= j < N_WAVE/2 */
			_ASSERT(j >= 0);
			_ASSERT(j < N_WAVE/2);
			wr =  Sinewave[j+N_WAVE/4];
			wi = -Sinewave[j];
			if(inverse)
				wi = -wi;
			if(shift) {
				wr >>= 1;
				wi >>= 1;
			}
			for(i=m; i<n; i+=istep) {
				j = i + l;
				tr = FIX_MPY(wr,fr[j]) - FIX_MPY(wi,fi[j]);
				ti = FIX_MPY(wr,fi[j]) + FIX_MPY(wi,fr[j]);
				qr = fr[i];
				qi = fi[i];
				if (shift) {
					qr >>= 1;
					qi >>= 1;
				}
				fr[j] = qr - tr;
				fi[j] = qi - ti;
				fr[i] = qr + tr;
				fi[i] = qi + ti;
			}
		}
		--k;
		l = istep;
    }

    return scale;
}

int fix_rfft(fixed fr[], fixed fi[], int m)
{
	int mr,nn,i,j,l,k,istep, n;
	fixed qr,qi,tr,ti,wr,wi, wr2, wi2;

	n = 1 << m;

	mr = 0;
	nn = n - 1;

	/* decimation in time - re-order data */
	for (m = 1; m <= nn; ++ m) {
		l = n;
		do {
			l >>= 1;
		} while (mr+l > nn);
		mr = (mr & (l-1)) + l;
		if (mr <= m) 
			continue;
		tr		= fr[m];
		fr[m]	= fr[mr];
		fr[mr]	= tr;
	}

	// first stage - radix 4 real butterflies
	wr  =  Sinewave[N_WAVE / 4] >> 1;
	wi  = -Sinewave[0] >> 1;
	wr2 =  Sinewave[(1 << (LOG2_N_WAVE - 2)) + N_WAVE/4] >> 1;
	wi2 = -Sinewave[1 << (LOG2_N_WAVE - 2)] >> 1;
	for (i = 0; i < n; ) {
		fixed tr1, tr2, tr3, tr4, ti1, ti2, ti3, ti4;

		tr  = FIX_MPY(wr, fr[i + 1]);
		ti  = FIX_MPY(wi, fr[i + 1]);
		qr  = fr[i] >> 1;
		tr1 = (qr + tr) >> 1;
		ti1 = ti >> 1;
		tr2 = (qr - tr) >> 1;
		ti2 = - ti >> 1;

		tr  = FIX_MPY(wr, fr[i + 3]);
		ti  = FIX_MPY(wi, fr[i + 3]);
		qr  = fr[i + 2] >> 1;
		tr3 = qr + tr;
		ti3 = ti;
		tr4 = qr - tr;
		ti4 = - ti;

		tr  = FIX_MPY(wr, tr3)  - FIX_MPY(wi, ti3);
		ti  = FIX_MPY(wr, ti3)  + FIX_MPY(wi, tr3);
		qr  = FIX_MPY(wr2, tr4) - FIX_MPY(wi2, ti4);
		qi  = FIX_MPY(wr2, ti4) + FIX_MPY(wi2, tr4);

		fr[i] = tr1 + tr;
		fi[i] = ti1 + ti;
		++ i;
		fr[i] = tr2 + qr;
		fi[i] = ti2 + qi;
		++ i;
		fr[i] = tr1 - tr;
		fi[i] = ti1 - ti;
		++ i;
		fr[i] = tr2 - qr;
		fi[i] = ti2 - qi;
		++ i;
	}

	k = LOG2_N_WAVE - 3;
	l = 4;
    while (l < n) {
		/* fixed scaling, for proper normalization -
			there will be log2(n) passes, so this
			results in an overall factor of 1/n,
			distributed to maximize arithmetic accuracy. */
		/* it may not be obvious, but the shift will be performed
			on each data point exactly once, during this pass. */
		istep = l << 1;
		for (m=0; m<l; ++m) {
			j = m << k;
			/* 0 <= j < N_WAVE/2 */
			_ASSERT(j >= 0);
			_ASSERT(j < N_WAVE/2);
			wr =  Sinewave[j+N_WAVE/4] >> 1;
			wi = -Sinewave[j] >> 1;
			for (i = m; i < n; i += istep) {
				j = i + l;
				tr = FIX_MPY(wr,fr[j]) - FIX_MPY(wi,fi[j]);
				ti = FIX_MPY(wr,fi[j]) + FIX_MPY(wi,fr[j]);
				qr = fr[i] >> 1;
				qi = fi[i] >> 1;
				fr[j] = qr - tr;
				fi[j] = qi - ti;
				fr[i] = qr + tr;
				fi[i] = qi + ti;
			}
		}
		-- k;
		l = istep;
    }

    return 0;
}

/*      window() - apply a Hanning window       */
void window(fixed fr[], int n)
{
    int i,j,k;

    j = N_WAVE/n;
    n >>= 1;
	for(i=0,k=N_WAVE/4; i<n; ++i,k+=j)
		fr[i] = FIX_MPY(fr[i], 16384-(Sinewave[k]>>1));
    n <<= 1;
	for(k-=j; i<n; ++i,k-=j)
        fr[i] = FIX_MPY(fr[i], 16384-(Sinewave[k]>>1));
}

/*      fix_loud() - compute loudness of freq-spectrum components.
        n should be ntot/2, where ntot was passed to fix_fft();
        6 dB is added to account for the omitted alias components.
        scale_shift should be the result of fix_fft(), if the time-series
        was obtained from an inverse FFT, 0 otherwise.
        loud[] is the loudness, in dB wrt 32767; will be +10 to -N_LOUD.
*/
void fix_loud(fixed loud[], fixed fr[], fixed fi[], int n, int scale_shift)
{
    int i, max;

    max = 0;
    if(scale_shift > 0)
        max = 10;
    scale_shift = (scale_shift+1) * 6;

    for(i=0; i<n; ++i) {
        loud[i] = db_from_ampl(fr[i],fi[i]) + scale_shift;
        if(loud[i] > max)
            loud[i] = max;
    }
}

/*      db_from_ampl() - find loudness (in dB) from
        the complex amplitude.
*/
int db_from_ampl(fixed re, fixed im)
{
    static long loud2[N_LOUD] = {0};
    long v;
    int i;

    if(loud2[0] == 0) {
        loud2[0] = (long)Loudampl[0] * (long)Loudampl[0];
        for(i=1; i<N_LOUD; ++i) {
            v = (long)Loudampl[i] * (long)Loudampl[i];
            loud2[i] = v;
            loud2[i-1] = (loud2[i-1]+v) / 2;
        }
    }

    v = (long)re * (long)re + (long)im * (long)im;

    for(i=0; i<N_LOUD; ++i)
        if(loud2[i] <= v)
            break;

    return (-i);
}

// decibel table << 16 from 1/2 to 1
static int aDbTable[64] = {
	-197283,
	-192870,
	-188525,
	-184245,
	-180028,
	-175873,
	-171778,
	-167740,
	-163760,
	-159834,
	-155961,
	-152141,
	-148371,
	-144651,
	-140978,
	-137352,
	-133772,
	-130236,
	-126744,
	-123294,
	-119885,
	-116517,
	-113188,
	-109898,
	-106645,
	-103429,
	-100249,
	-97104,
	-93993,
	-90916,
	-87872,
	-84860,
	-81880,
	-78930,
	-76011,
	-73122,
	-70261,
	-67429,
	-64625,
	-61848,
	-59098,
	-56375,
	-53677,
	-51004,
	-48357,
	-45733,
	-43134,
	-40558,
	-38006,
	-35476,
	-32968,
	-30482,
	-28018,
	-25575,
	-23153,
	-20751,
	-18369,
	-16007,
	-13664,
	-11341,
	-9036,
	-6750,
	-4482,
	-2232
};

int db_from_ampl2(fixed re, fixed im)
{
	// compute square amplitude of the input signal
	UINT amp2     = re * re + im * im;
	int  exponent;
	int  res;

	if (amp2 == 0)
		return 0;

	//FIXME
//	return (int)(10.0 * 65536.0 * log10(re * re + im * im) + 0.5);

	// normalize amp2
	// exponent needs 5 bits (0..30)
	exponent = 31;
	amp2 <<= 1;
	while (exponent && !(amp2 & 0x80000000)) {
		-- exponent; 
		amp2 <<= 1;
	}
	res = exponent * 197283;

	// LOG(N)=LOG(M * 2^E)=LOG(M) + E*LOG(2)

	// search mantisa in the lookup table
	res += aDbTable[(amp2&0x7FFFFFFF)>>(31 - 6)];
	return res;
}

int db_from_ampl2_2(UINT amp2)
{
	// compute square amplitude of the input signal
	int  exponent;
	int  res;

	if (amp2 == 0)
		return 0;

	//FIXME
//	return (int)(10.0 * 65536.0 * log10(re * re + im * im) + 0.5);

	// normalize amp2
	// exponent needs 5 bits (0..30)
	exponent = 31;
	amp2 <<= 1;
	while (exponent && !(amp2 & 0x80000000)) {
		-- exponent; 
		amp2 <<= 1;
	}
	res = exponent * 197283;

	// LOG(N)=LOG(M * 2^E)=LOG(M) + E*LOG(2)

	// search mantisa in the lookup table
	res += aDbTable[(amp2&0x7FFFFFFF)>>(31 - 6)];
	return res;
}

double db_from_ampl2_test()
{
	double diffMax = 0.0;
	int    i, j;

	for (i = 100; i < 32767; ++ i) {
		for (j = 100; j < 32767; j += 100) {
			double db1 = (double)db_from_ampl2(i, j);
			double db2 = 10.0 * 65536.0 * log10(i * i + j * j);
			double diff = fabs(db2 - db1);
			if (diff > diffMax)
				diffMax = diff;
		}
	}

	return diffMax;
}

/*
        iscale() - scale an integer value by (numer/denom)
*/
int iscale(int value, int numer, int denom)
{
#ifdef  DOS
        asm     mov ax,value
        asm     imul WORD PTR numer
        asm     idiv WORD PTR denom

        return _AX;
#else
        return (long) value * (long)numer/(long)denom;
#endif
}

/*
#if N_WAVE != 1024
        ERROR: N_WAVE != 1024
#endif
fixed Sinewave[1024] = {
      0,    201,    402,    603,    804,   1005,   1206,   1406,
   1607,   1808,   2009,   2209,   2410,   2610,   2811,   3011,
   3211,   3411,   3611,   3811,   4011,   4210,   4409,   4608,
   4807,   5006,   5205,   5403,   5601,   5799,   5997,   6195,
   6392,   6589,   6786,   6982,   7179,   7375,   7571,   7766,
   7961,   8156,   8351,   8545,   8739,   8932,   9126,   9319,
   9511,   9703,   9895,  10087,  10278,  10469,  10659,  10849,
  11038,  11227,  11416,  11604,  11792,  11980,  12166,  12353,
  12539,  12724,  12909,  13094,  13278,  13462,  13645,  13827,
  14009,  14191,  14372,  14552,  14732,  14911,  15090,  15268,
  15446,  15623,  15799,  15975,  16150,  16325,  16499,  16672,
  16845,  17017,  17189,  17360,  17530,  17699,  17868,  18036,
  18204,  18371,  18537,  18702,  18867,  19031,  19194,  19357,
  19519,  19680,  19840,  20000,  20159,  20317,  20474,  20631,
  20787,  20942,  21096,  21249,  21402,  21554,  21705,  21855,
  22004,  22153,  22301,  22448,  22594,  22739,  22883,  23027,
  23169,  23311,  23452,  23592,  23731,  23869,  24006,  24143,
  24278,  24413,  24546,  24679,  24811,  24942,  25072,  25201,
  25329,  25456,  25582,  25707,  25831,  25954,  26077,  26198,
  26318,  26437,  26556,  26673,  26789,  26905,  27019,  27132,
  27244,  27355,  27466,  27575,  27683,  27790,  27896,  28001,
  28105,  28208,  28309,  28410,  28510,  28608,  28706,  28802,
  28897,  28992,  29085,  29177,  29268,  29358,  29446,  29534,
  29621,  29706,  29790,  29873,  29955,  30036,  30116,  30195,
  30272,  30349,  30424,  30498,  30571,  30643,  30713,  30783,
  30851,  30918,  30984,  31049,
  31113,  31175,  31236,  31297,
  31356,  31413,  31470,  31525,  31580,  31633,  31684,  31735,
  31785,  31833,  31880,  31926,  31970,  32014,  32056,  32097,
  32137,  32176,  32213,  32249,  32284,  32318,  32350,  32382,
  32412,  32441,  32468,  32495,  32520,  32544,  32567,  32588,
  32609,  32628,  32646,  32662,  32678,  32692,  32705,  32717,
  32727,  32736,  32744,  32751,  32757,  32761,  32764,  32766,
  32767,  32766,  32764,  32761,  32757,  32751,  32744,  32736,
  32727,  32717,  32705,  32692,  32678,  32662,  32646,  32628,
  32609,  32588,  32567,  32544,  32520,  32495,  32468,  32441,
  32412,  32382,  32350,  32318,  32284,  32249,  32213,  32176,
  32137,  32097,  32056,  32014,  31970,  31926,  31880,  31833,
  31785,  31735,  31684,  31633,  31580,  31525,  31470,  31413,
  31356,  31297,  31236,  31175,  31113,  31049,  30984,  30918,
  30851,  30783,  30713,  30643,  30571,  30498,  30424,  30349,
  30272,  30195,  30116,  30036,  29955,  29873,  29790,  29706,
  29621,  29534,  29446,  29358,  29268,  29177,  29085,  28992,
  28897,  28802,  28706,  28608,  28510,  28410,  28309,  28208,
  28105,  28001,  27896,  27790,  27683,  27575,  27466,  27355,
  27244,  27132,  27019,  26905,  26789,  26673,  26556,  26437,
  26318,  26198,  26077,  25954,  25831,  25707,  25582,  25456,
  25329,  25201,  25072,  24942,  24811,  24679,  24546,  24413,
  24278,  24143,  24006,  23869,  23731,  23592,  23452,  23311,
  23169,  23027,  22883,  22739,  22594,  22448,  22301,  22153,
  22004,  21855,  21705,  21554,  21402,  21249,  21096,  20942,
  20787,  20631,  20474,  20317,  20159,  20000,  19840,  19680,
  19519,  19357,  19194,  19031,  18867,  18702,  18537,  18371,
  18204,  18036,  17868,  17699,  17530,  17360,  17189,  17017,
  16845,  16672,  16499,  16325,  16150,  15975,  15799,  15623,
  15446,  15268,  15090,  14911,  14732,  14552,  14372,  14191,
  14009,  13827,  13645,  13462,  13278,  13094,  12909,  12724,
  12539,  12353,  12166,  11980,  11792,  11604,  11416,  11227,
  11038,  10849,  10659,  10469,  10278,  10087,   9895,   9703,
   9511,   9319,   9126,   8932,   8739,   8545,   8351,   8156,
   7961,   7766,   7571,   7375,   7179,   6982,   6786,   6589,
   6392,   6195,   5997,   5799,   5601,   5403,   5205,   5006,
   4807,   4608,   4409,   4210,   4011,   3811,   3611,   3411,
   3211,   3011,   2811,   2610,   2410,   2209,   2009,   1808,
   1607,   1406,   1206,   1005,    804,    603,    402,    201,
      0,   -201,   -402,   -603,   -804,  -1005,  -1206,  -1406,
  -1607,  -1808,  -2009,  -2209,  -2410,  -2610,  -2811,  -3011,
  -3211,  -3411,  -3611,  -3811,  -4011,  -4210,  -4409,  -4608,
  -4807,  -5006,  -5205,  -5403,  -5601,  -5799,  -5997,  -6195,
  -6392,  -6589,  -6786,  -6982,  -7179,  -7375,  -7571,  -7766,
  -7961,  -8156,  -8351,  -8545,  -8739,  -8932,  -9126,  -9319,
  -9511,  -9703,  -9895, -10087, -10278, -10469, -10659, -10849,
 -11038, -11227, -11416, -11604, -11792, -11980, -12166, -12353,
 -12539, -12724, -12909, -13094, -13278, -13462, -13645, -13827,
 -14009, -14191, -14372, -14552, -14732, -14911, -15090, -15268,
 -15446, -15623, -15799, -15975, -16150, -16325, -16499, -16672,
 -16845, -17017, -17189, -17360, -17530, -17699, -17868, -18036,
 -18204, -18371, -18537, -18702, -18867, -19031, -19194, -19357,
 -19519, -19680, -19840, -20000, -20159, -20317, -20474, -20631,
 -20787, -20942, -21096, -21249, -21402, -21554, -21705, -21855,
 -22004, -22153, -22301, -22448, -22594, -22739, -22883, -23027,
 -23169, -23311, -23452, -23592, -23731, -23869, -24006, -24143,
 -24278, -24413, -24546, -24679, -24811, -24942, -25072, -25201,
 -25329, -25456, -25582, -25707, -25831, -25954, -26077, -26198,
 -26318, -26437, -26556, -26673, -26789, -26905, -27019, -27132,
 -27244, -27355, -27466, -27575, -27683, -27790, -27896, -28001,
 -28105, -28208, -28309, -28410, -28510, -28608, -28706, -28802,
 -28897, -28992, -29085, -29177, -29268, -29358, -29446, -29534,
 -29621, -29706, -29790, -29873, -29955, -30036, -30116, -30195,
 -30272, -30349, -30424, -30498, -30571, -30643, -30713, -30783,
 -30851, -30918, -30984, -31049, -31113, -31175, -31236, -31297,
 -31356, -31413, -31470, -31525, -31580, -31633, -31684, -31735,
 -31785, -31833, -31880, -31926, -31970, -32014, -32056, -32097,
 -32137, -32176, -32213, -32249, -32284, -32318, -32350, -32382,
 -32412, -32441, -32468, -32495, -32520, -32544, -32567, -32588,
 -32609, -32628, -32646, -32662, -32678, -32692, -32705, -32717,
 -32727, -32736, -32744, -32751, -32757, -32761, -32764, -32766,
 -32767, -32766, -32764, -32761, -32757, -32751, -32744, -32736,
 -32727, -32717, -32705, -32692, -32678, -32662, -32646, -32628,
 -32609, -32588, -32567, -32544, -32520, -32495, -32468, -32441,
 -32412, -32382, -32350, -32318, -32284, -32249, -32213, -32176,
 -32137, -32097, -32056, -32014, -31970, -31926, -31880, -31833,
 -31785, -31735, -31684, -31633, -31580, -31525, -31470, -31413,
 -31356, -31297, -31236, -31175, -31113, -31049, -30984, -30918,
 -30851, -30783, -30713, -30643, -30571, -30498, -30424, -30349,
 -30272, -30195, -30116, -30036, -29955, -29873, -29790, -29706,
 -29621, -29534, -29446, -29358, -29268, -29177, -29085, -28992,
 -28897, -28802, -28706, -28608, -28510, -28410, -28309, -28208,
 -28105, -28001, -27896, -27790, -27683, -27575, -27466, -27355,
 -27244, -27132, -27019, -26905, -26789, -26673, -26556, -26437,
 -26318, -26198, -26077, -25954, -25831, -25707, -25582, -25456,
 -25329, -25201, -25072, -24942, -24811, -24679, -24546, -24413,
 -24278, -24143, -24006, -23869, -23731, -23592, -23452, -23311,
 -23169, -23027, -22883, -22739, -22594, -22448, -22301, -22153,
 -22004, -21855, -21705, -21554, -21402, -21249, -21096, -20942,
 -20787, -20631, -20474, -20317, -20159, -20000, -19840, -19680,
 -19519, -19357, -19194, -19031, -18867, -18702, -18537, -18371,
 -18204, -18036, -17868, -17699, -17530, -17360, -17189, -17017,
 -16845, -16672, -16499, -16325, -16150, -15975, -15799, -15623,
 -15446, -15268, -15090, -14911, -14732, -14552, -14372, -14191,
 -14009, -13827, -13645, -13462, -13278, -13094, -12909, -12724,
 -12539, -12353, -12166, -11980, -11792, -11604, -11416, -11227,
 -11038, -10849, -10659, -10469, -10278, -10087,  -9895,  -9703,
  -9511,  -9319,  -9126,  -8932,  -8739,  -8545,  -8351,  -8156,
  -7961,  -7766,  -7571,  -7375,  -7179,  -6982,  -6786,  -6589,
  -6392,  -6195,  -5997,  -5799,  -5601,  -5403,  -5205,  -5006,
  -4807,  -4608,  -4409,  -4210,  -4011,  -3811,  -3611,  -3411,
  -3211,  -3011,  -2811,  -2610,  -2410,  -2209,  -2009,  -1808,
  -1607,  -1406,  -1206,  -1005,   -804,   -603,   -402,   -201,
};
*/

#if N_LOUD != 100
        ERROR: N_LOUD != 100
#endif
fixed Loudampl[100] = {
  32767,  29203,  26027,  23197,  20674,  18426,  16422,  14636,
  13044,  11626,  10361,   9234,   8230,   7335,   6537,   5826,
   5193,   4628,   4125,   3676,   3276,   2920,   2602,   2319,
   2067,   1842,   1642,   1463,   1304,   1162,   1036,    923,
    823,    733,    653,    582,    519,    462,    412,    367,
    327,    292,    260,    231,    206,    184,    164,    146,
    130,    116,    103,     92,     82,     73,     65,     58,
     51,     46,     41,     36,     32,     29,     26,     23,
     20,     18,     16,     14,     13,     11,     10,      9,
      8,      7,      6,      5,      5,      4,      4,      3,
      3,      2,      2,      2,      2,      1,      1,      1,
      1,      1,      1,      0,      0,      0,      0,      0,
      0,      0,      0,      0,
};


/*
 * Real Fast Fourier Transform
 *
 * This function was based on the code in Numerical Recipes in C.
 * In Num. Rec., the inner loop is based on a single 1-based array
 * of interleaved real and imaginary numbers.  Because we have two
 * separate zero-based arrays, our indices are quite different.
 * Here is the correspondence between Num. Rec. indices and our indices:
 *
 * i1  <->  real[i]
 * i2  <->  imag[i]
 * i3  <->  real[n/2-i]
 * i4  <->  imag[n/2-i]
 */

/*
void RealFFT(int NumSamples, float *RealIn, float *RealOut, float *ImagOut)
{
   int Half = NumSamples / 2;
   int i;

   float theta = M_PI / Half;

   float *tmpReal = new float[Half];
   float *tmpImag = new float[Half];

   for (i = 0; i < Half; i++) {
      tmpReal[i] = RealIn[2 * i];
      tmpImag[i] = RealIn[2 * i + 1];
   }

   FFT(Half, 0, tmpReal, tmpImag, RealOut, ImagOut);

   float wtemp = float (sin(0.5 * theta));

   float wpr = -2.0 * wtemp * wtemp;
   float wpi = float (sin(theta));
   float wr = 1.0 + wpr;
   float wi = wpi;

   int i3;

   float h1r, h1i, h2r, h2i;

   for (i = 1; i < Half / 2; i++) {

      i3 = Half - i;

      h1r = 0.5 * (RealOut[i] + RealOut[i3]);
      h1i = 0.5 * (ImagOut[i] - ImagOut[i3]);
      h2r = 0.5 * (ImagOut[i] + ImagOut[i3]);
      h2i = -0.5 * (RealOut[i] - RealOut[i3]);

      RealOut[i] = h1r + wr * h2r - wi * h2i;
      ImagOut[i] = h1i + wr * h2i + wi * h2r;
      RealOut[i3] = h1r - wr * h2r + wi * h2i;
      ImagOut[i3] = -h1i + wr * h2i + wi * h2r;

      wr = (wtemp = wr) * wpr - wi * wpi + wr;
      wi = wi * wpr + wtemp * wpi + wi;
   }

   RealOut[0] = (h1r = RealOut[0]) + ImagOut[0];
   ImagOut[0] = h1r - ImagOut[0];

   delete[]tmpReal;
   delete[]tmpImag;
}
*/

/*
 * PowerSpectrum
 *
 * This function computes the same as RealFFT, above, but
 * adds the squares of the real and imaginary part of each
 * coefficient, extracting the power and throwing away the
 * phase.
 *
 * For speed, it does not call RealFFT, but duplicates some
 * of its code.
 */

/*
void PowerSpectrum(int NumSamples, float *In, float *Out)
{
   int Half = NumSamples / 2;
   int i;

   float theta = M_PI / Half;

   float *tmpReal = new float[Half];
   float *tmpImag = new float[Half];
   float *RealOut = new float[Half];
   float *ImagOut = new float[Half];

   for (i = 0; i < Half; i++) {
      tmpReal[i] = In[2 * i];
      tmpImag[i] = In[2 * i + 1];
   }

   FFT(Half, 0, tmpReal, tmpImag, RealOut, ImagOut);

   float wtemp = float (sin(0.5 * theta));

   float wpr = -2.0 * wtemp * wtemp;
   float wpi = float (sin(theta));
   float wr = 1.0 + wpr;
   float wi = wpi;

   int i3;

   float h1r, h1i, h2r, h2i, rt, it;

   for (i = 1; i < Half / 2; i++) {

      i3 = Half - i;

      h1r = 0.5 * (RealOut[i] + RealOut[i3]);
      h1i = 0.5 * (ImagOut[i] - ImagOut[i3]);
      h2r = 0.5 * (ImagOut[i] + ImagOut[i3]);
      h2i = -0.5 * (RealOut[i] - RealOut[i3]);

      rt = h1r + wr * h2r - wi * h2i;
      it = h1i + wr * h2i + wi * h2r;

      Out[i] = rt * rt + it * it;

      rt = h1r - wr * h2r + wi * h2i;
      it = -h1i + wr * h2i + wi * h2r;

      Out[i3] = rt * rt + it * it;

      wr = (wtemp = wr) * wpr - wi * wpi + wr;
      wi = wi * wpr + wtemp * wpi + wi;
   }

   rt = (h1r = RealOut[0]) + ImagOut[0];
   it = h1r - ImagOut[0];
   Out[0] = rt * rt + it * it;

   rt = RealOut[Half / 2];
   it = ImagOut[Half / 2];
   Out[Half / 2] = rt * rt + it * it;

   delete[]tmpReal;
   delete[]tmpImag;
   delete[]RealOut;
   delete[]ImagOut;
}
*/

/*
Here is a C function for doing a real fft (split radix).  It is a translation
and correction of some Fortran code.  I have tested it somewhat, but maybe
you could test it yourself (and see how it compares to others).

Below is the calling program, followed by rfft().

Currently it uses Fortran style: x[1] is first element not x[0].  I couldn't
figure out a simple way to convert to C style.  I guess it's not a 
catastrophe to waste one element (x[0]).

Cheers,
Bill Simpson
*/

/****************************************************************************
* rfft(float X[],int N)                                                     *
*     A real-valued, in-place, split-radix FFT program                      *
*     Decimation-in-time, cos/sin in second loop                            *
*     Input: float X[1]...X[N] (NB Fortran style: 1st pt X[1] not X[0]!)    *
*     Length is N=2**M (i.e. N must be power of 2--no error checking)       *
*     Output in X[1]...X[N], in order:                                      *
*           [Re(0), Re(1),..., Re(N/2), Im(N/2-1),..., Im(1)]               *
*                                                                           *
* Original Fortran code by Sorensen; published in H.V. Sorensen, D.L. Jones,*
* M.T. Heideman, C.S. Burrus (1987) Real-valued fast fourier transform      *
* algorithms.  IEEE Trans on Acoustics, Speech, & Signal Processing, 35,    *
* 849-863.  Adapted to C by Bill Simpson, 1995  wsimpson@uwinnipeg.ca       *
****************************************************************************/

void rfft(float X[],int N)
{
	int I,I0,I1,I2,I3,I4,I5,I6,I7,I8, IS,ID;
	int J,K,M,N2,N4,N8;
	float A,A3,CC1,SS1,CC3,SS3,E,R1,XT;
	float T1,T2,T3,T4,T5,T6;  

	M=(int)(log(N)/log(2.0));               /* N=2^M */

	/* ----Digit reverse counter--------------------------------------------- */
	J = 1;
	for(I=1;I<N;I++) {
		if (I<J) {
			XT    = X[J];
			X[J]  = X[I];
			X[I]  = XT;
        }
        K = N/2;
		while(K<J) {
			J -= K;
			K /= 2;
        }
        J += K;
	}

	/* ----Length two butterflies--------------------------------------------- */
	IS = 1;
	ID = 4;
	do {
		for(I0 = IS;I0<=N;I0+=ID) {
            I1    = I0 + 1;
            R1    = X[I0];
            X[I0] = R1 + X[I1];
            X[I1] = R1 - X[I1];
        }
        IS = 2 * ID - 1;
        ID = 4 * ID;
   } while (IS < N);


	/* ----L shaped butterflies----------------------------------------------- */
	N2 = 2;
	for(K=2;K<=M;K++) {
        N2    = N2 * 2;
        N4    = N2/4;
        N8    = N2/8;
        E     = (float) 6.2831853071719586f/N2;
        IS    = 0;
        ID    = N2 * 2;
        do {
			for(I=IS;I<N;I+=ID) {
                I1 = I + 1;
                I2 = I1 + N4;
                I3 = I2 + N4;
                I4 = I3 + N4;
                T1 = X[I4] +X[I3];
                X[I4] = X[I4] - X[I3];
                X[I3] = X[I1] - T1;
                X[I1] = X[I1] + T1;
                if(N4!=1) {
                    I1 += N8;
                    I2 += N8;
                    I3 += N8;
                    I4 += N8;
                    T1 = (X[I3] + X[I4])*.7071067811865475244f;
                    T2 = (X[I3] - X[I4])*.7071067811865475244f;
                    X[I4] = X[I2] - T1;
                    X[I3] = -X[I2] - T1;
                    X[I2] = X[I1] - T2;
                    X[I1] = X[I1] + T2;
                }
            }
            IS = 2 * ID - N2;
            ID = 4 * ID;
        } while(IS<N);
        A = E;
        for(J= 2;J<=N8;J++)
                {
                A3	  = (float)3.0 * A;
                CC1   = (float)cos(A);
                SS1   = (float)sin(A);  /*typo A3--really A?*/
                CC3   = (float)cos(A3); /*typo 3--really A3?*/
                SS3   = (float)sin(A3);
                A = (float)J * E;
                IS = 0;
                ID = 2 * N2;
                do        
                        {
                        for(I=IS;I<N;I+=ID)
                                {
                                I1 = I + J;
                                I2 = I1 + N4;
                                I3 = I2 + N4;
                                I4 = I3 + N4;
                                I5 = I + N4 - J + 2;
                                I6 = I5 + N4;
                                I7 = I6 + N4;
                                I8 = I7 + N4;
                                T1 = X[I3] * CC1 + X[I7] * SS1;
                                T2 = X[I7] * CC1 - X[I3] * SS1;
                                T3 = X[I4] * CC3 + X[I8] * SS3;
                                T4 = X[I8] * CC3 - X[I4] * SS3;
                                T5 = T1 + T3;
                                T6 = T2 + T4;
                                T3 = T1 - T3;
                                T4 = T2 - T4;
                                T2 = X[I6] + T6;
                                X[I3] = T6 - X[I6];
                                X[I8] = T2;
                                T2    = X[I2] - T3;
                                X[I7] = -X[I2] - T3;
                                X[I4] = T2;
                                T1    = X[I1] + T5;
                                X[I6] = X[I1] - T5;
                                X[I1] = T1;
                                T1    = X[I5] + T4;
                                X[I5] = X[I5] - T4;
                                X[I2] = T1;
                                }
                        IS = 2 * ID - N2;
                        ID = 4 * ID;
                        }while(IS<N);
                }
        }
}

/////////////////////////////////////////////////////////
// Sorensen in-place radix-2 FFT for real values
// data: array of doubles:
// re(0),re(1),re(2),...,re(size-1)
// 
// output:
// re(0),re(1),re(2),...,re(size/2),im(size/2-1),...,im(1)
// normalized by array length
//
// Source: 
// Sorensen et al: Real-Valued Fast Fourier Transform Algorithms,
// IEEE Trans. ASSP, ASSP-35, No. 6, June 1987

void sorensen_realfft_radix2(double *data,long n)
{
    double  xt,a,e, t1, t2, cc, ss;
    long  i, j, k, n1, n2, n3, n4, i1, i2, i3, i4;

	n4=n-1;
  //data shuffling
      for (i=0,j=0,n2=n/2; i<n4 ; i++){
	  if (i<j){
				xt=data[j];
				data[j]=data[i];
				data[i]=xt;
				}
	  k=n2;
	  while (k<=j){
				j-=k;
				k>>=1;	
				}
	  j+=k;
      }
	
/* -------------------- */
    for (i=0; i<n; i += 2)  
      {
	 xt = data[i];
	 data[i] = xt + data[i+1];
	 data[i+1] = xt - data[i+1];
      }
/* ------------------------ */
    n2 = 1;
    for (k=n;k>2;k>>=1){ 
		n4 = n2;
		n2 = n4 << 1;
		n1 = n2 << 1;
		e = 2*3.1415926535897932384626433832795/(n1);
		for (i=0; i<n; i+=n1){  
			xt = data[i];
			data[i] = xt + data[i+n2];
			data[i+n2] = xt-data[i+n2];
			data[i+n4+n2] = -data[i+n4+n2];
			a = e;
			n3=n4-1;
			for (j = 1; j <=n3; j++){
				i1 = i+j;
				i2 = i - j + n2;
				i3 = i1 + n2;
				i4 = i - j + n1;
				cc = cos(a);
				ss = sin(a);
				a += e;
				t1 = data[i3] * cc + data[i4] * ss;
				t2 = data[i3] * ss - data[i4] * cc;
				data[i4] = data[i2] - t2;
				data[i3] = -data[i2] - t2;
				data[i2] = data[i1] - t1;
				data[i1] += t1;
		  }
	  }
  }
	
	//division with array length
   for(i=0;i<n;i++) data[i]/=n;
}