/*
 *    filter.c  --  FIR filter
 *
 *    Copyright (C) 2001, 2002, 2003
 *      Tomi Manninen (oh2bns@sral.fi)
 *
 *    This file is part of gMFSK.
 *
 *    gMFSK is free software; you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation; either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    gMFSK is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with gMFSK; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

#include <windows.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>

#include "cmplx.h"
#include "filter.h"

#undef	DEBUG

#ifdef DEBUG
#include <stdio.h>
#endif

int mac(const int *a, const int *b, unsigned int len)
{ 
	int sum = 0;
	unsigned int i;

	for (i = 0; i < len; i++)
		sum += (*a++) * (*b++);
	return sum;
}

/* ---------------------------------------------------------------------- */

/*
 * Create a band pass FIR filter with 6 dB corner frequencies
 * of 'f1' and 'f2'. (0 <= f1 < f2 <= 0.5)
 */
static float *mk_filter(int len, int hilbert, double f1, double f2, BOOL bNormalize)
{
	double *dblfir	= (double*)malloc(sizeof(double) * len);
	float  *fir		= (float*)malloc(sizeof(float) * len);
	double  sum		= 0.0;
	int		i;

	for (i = 0; i < len; ++ i) {
		double t = (double)i - 0.5 * ((double)len - 1.0);
		double h = (double)i / ((double)len - 1.0);
		double x = hilbert ? 
			/*
			 * The actual filter code assumes the impulse response
			 * is in time reversed order. This will be anti-
			 * symmetric so the minus sign handles that for us.
			 */
			(f1 * cosc(2.0 * f1 * t) - f2 * cosc(2.0 * f2 * t)) :
			(f2 * sinc(2.0 * f2 * t) - f1 * sinc(2.0 * f1 * t));
		x = 2.0 * x * hamming(h);
		sum += fabs(x);
		dblfir[i] = x;
	}

	// normalize the filter
	if (bNormalize) {
		for (i = 0; i < len; ++ i)
			fir[i] = (float)(dblfir[i] / sum);
	} else {
		for (i = 0; i < len; ++ i)
			fir[i] = (float)dblfir[i];
	}

	free(dblfir);
	return fir;
}

struct filter *filter_init(int len, int dec, float *itaps, float *qtaps)
{
	struct filter *f;
	int            i;

	if ((f = calloc(1, sizeof(struct filter))) == NULL)
		return NULL;

	f->length = len;
	f->decimateratio = dec;

	if (itaps) {
		//FIXME
		double dSum = 0.0;
		if ((f->ifilter = malloc(len * sizeof(int))) == NULL) {
			filter_free(f);
			return NULL;
		}
		for (i = 0; i < len; ++ i) {
			double dValue = (double)itaps[i] * (double)0x7fff;
			int    iValue;
			if (dValue > 0.0) {
				iValue = (int)floor(dValue + 0.5);
				if (iValue > 0x7fff)
					iValue = 0x7fff;
			} else {
				iValue = (int)floor(dValue - 0.5);
				if (iValue < -0x7fff)
					iValue = -0x7fff;
			}
			f->ifilter[i] =  iValue;
			dSum += iValue;
		}
		printf("dSum: %f\n", dSum);
	}

	if (qtaps) {
		//FIXME
		double dSum = 0.0;
		if ((f->qfilter = malloc(len * sizeof(int))) == NULL) {
			filter_free(f);
			return NULL;
		}
		for (i = 0; i < len; ++ i) {
			double dValue = (double)qtaps[i] * (double)0x7fff;
			int    iValue;
			if (dValue > 0.0) {
				iValue = (int)floor(dValue + 0.5);
				if (iValue > 0x7fff)
					iValue = 0x7fff;
			} else {
				iValue = (int)ceil(dValue - 0.5);
				if (iValue < -0x7fff)
					iValue = -0x7fff;
			}
			f->qfilter[i] =  iValue;
			dSum += iValue;
		}
		printf("dSum: %f\n", dSum);
	}

	f->pointer = len;
	f->counter = 0;

	return f;
}

struct filter *filter_init_lowpass(int len, int dec, float freq)
{
	struct filter *f;
	float *i, *q;

	i = mk_filter(len, 0, 0.0, (double)freq, TRUE);
	q = (float*)malloc(sizeof(float) * len);
	memcpy(q, i, sizeof(float) * len);

	f = filter_init(len, dec, i, q);

	free(i);
	free(q);

	return f;
}

struct filter *filter_init_bandpass(int len, int dec, float f1, float f2)
{
	struct filter *f;
	float *i, *q;

	i = mk_filter(len, 0, (double)f1, (double)f2, TRUE);
	q = (float*)malloc(sizeof(float) * len);
	memcpy(q, i, sizeof(float) * len);

	f = filter_init(len, dec, i, q);

	free(i);
	free(q);

	return f;
}

struct filter *filter_init_hilbert(int len, int dec)
{
	struct filter *f;
	float *iCoeff, *qCoeff;
	float  iSum = 0.0f;
	float  qSum = 0.0f;
	int    i;

	iCoeff = mk_filter(len, 0, 0.05, 0.45, FALSE);
	qCoeff = mk_filter(len, 1, 0.05, 0.45, FALSE);

	// normalize the filters
	for (i = 0; i < len; ++ i) {
		iSum += fabs(iCoeff[i]);
		qSum += fabs(qCoeff[i]);
	}
	if (qSum > iSum)
		iSum = qSum;
	for (i = 0; i < len; ++ i) {
		iCoeff[i] /= iSum;
		qCoeff[i] /= iSum;
	}

	f = filter_init(len, dec, iCoeff, qCoeff);

	free(iCoeff);
	free(qCoeff);

	return f;
}

void filter_free(struct filter *f)
{
	if (f) {
		free(f->ifilter);
		free(f->qfilter);
		free(f);
	}
}

int filter_run(struct filter *f, complex in, complex *out)
{
	int *iptr = f->ibuffer + f->pointer;
	int *qptr = f->qbuffer + f->pointer;

	++ f->pointer;
	++ f->counter;

	*iptr = c_re(in);
	*qptr = c_im(in);

	if (f->counter == f->decimateratio) {
		out->re = mac(iptr - f->length, f->ifilter, f->length);
		out->im = mac(qptr - f->length, f->qfilter, f->length);
	}

	if (f->pointer == BufferLen) {
		iptr = f->ibuffer + BufferLen - f->length;
		qptr = f->qbuffer + BufferLen - f->length;
		memcpy(f->ibuffer, iptr, f->length * sizeof(int));
		memcpy(f->qbuffer, qptr, f->length * sizeof(int));
		f->pointer = f->length;
	}

	if (f->counter == f->decimateratio) {
		f->counter = 0;
		return 1;
	}

	return 0;
}
