/*
 *    cmplx.h  --  Complex arithmetic
 *
 *    Copyright (C) 2001, 2002, 2003
 *      Tomi Manninen (oh2bns@sral.fi)
 *
 *    This file is part of gMFSK.
 *
 *    gMFSK is free software; you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation; either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    gMFSK is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with gMFSK; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

#ifndef _COMPLEX_H
#define _COMPLEX_H

#include <math.h>

#ifndef M_PI
#define M_PI (3.1415926535897932384626433832795)
#endif /* M_PI */

#ifndef M_PI_2
#define M_PI_2 (6.283185307179586476925286766559)
#endif /* M_PI_2 */

#ifndef M_SQRT2
#define M_SQRT2 1.41421356237309504880
#endif /* M_SQRT2 */

typedef struct {
     int re, im;
} complex;

#define c_re(c)  ((c).re)
#define c_im(c)  ((c).im)

#if 0
/*
 * Complex multiplication.
 */
extern __inline complex cmul(complex x, complex y)
{
	complex z;

	c_re(z) = c_re(x) * c_re(y) - c_im(x) * c_im(y);
	c_im(z) = c_re(x) * c_im(y) + c_im(x) * c_re(y);

	return z;
}

/*
 * Complex addition.
 */
extern __inline complex cadd(complex x, complex y)
{
	complex z;

	c_re(z) = c_re(x) + c_re(y);
	c_im(z) = c_im(x) + c_im(y);

	return z;
}

/*
 * Complex subtraction.
 */
extern __inline complex csub(complex x, complex y)
{
	complex z;

	c_re(z) = c_re(x) - c_re(y);
	c_im(z) = c_im(x) - c_im(y);

	return z;
}

/*
 * Complex multiply-accumulate.
 */
extern __inline complex cmac(complex *a, complex *b, int ptr, int len)
{
	complex z;
	int i;

	c_re(z) = 0;
	c_im(z) = 0;

	ptr = ptr % len;

	for (i = 0; i < len; i++) {
		z = cadd(z, cmul(a[i], b[ptr]));
		ptr = (ptr + 1) % len;
	}

	return z;
}

/*
 * Complex ... yeah, what??? Returns a complex number that has the
 * properties: |z| = |x| * |y|  and  arg(z) = arg(y) - arg(x)
 */
extern __inline complex ccor(complex x, complex y)
{
	complex z;

	c_re(z) = (c_re(x) >> 16) * (c_re(y) >> 16) + (c_im(x) >> 16) * (c_im(y) >> 16);
	c_im(z) = (c_re(x) >> 16) * (c_im(y) >> 16) - (c_im(x) >> 16) * (c_re(y) >> 16);

	return z;
}

/*
 * Real part of the complex ???
 */
extern __inline double ccorI(complex x, complex y)
{
	return c_re(x) * c_re(y) + c_im(x) * c_im(y);
}

/*
 * Imaginary part of the complex ???
 */
extern __inline double ccorQ(complex x, complex y)
{
	return c_re(x) * c_im(y) - c_im(x) * c_re(y);
}

/*
 * Modulo (absolute value) of a complex number.
 */
extern __inline double cmod(complex x)
{
	return isqrt(c_re(x) * c_re(x) + c_im(x) * c_im(x));
}

/*
 * Square of the absolute value (power).
 */
extern __inline double cpwr(complex x)
{
	return (c_re(x) * c_re(x) + c_im(x) * c_im(x));
}

/*
 * Argument of a complex number.
 */
extern __inline double carg(complex x)
{
	return atan2(c_im(x), c_re(x));
}
#endif 0

#endif
