/*
 *    mfskrx.c  --  MFSK demodulator
 *
 *    Copyright (C) 2001, 2002, 2003
 *      Tomi Manninen (oh2bns@sral.fi)
 *
 *    This file is part of gMFSK.
 *
 *    gMFSK is free software; you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation; either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    gMFSK is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with gMFSK; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

#include <windows.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>

#ifdef _WIN32_WCE
#define _ASSERT(x) 
#else
	#include <crtdbg.h>
#endif /* _WIN32_WCE */

#include "../misc/filter.h"
#include "../misc/misc.h"
#include "../misc/mixer.h"
#include "../misc/fixedpoint.h"

#include "mfsk.h"
#include "mfskvaricode.h"

// #include "../main/dbgwaterfall.h"

// extern DbgWaterfall s_DbgWaterfall;

//#include "picture.h"

/*
static void recvpic(Trx *trx, complex z)
{
	Mfsk *m = (Mfsk*)trx->modem;

	m->picf += carg(ccor(m->prevz, z)) * SampleRate / (2.0 * M_PI);
	m->prevz = z;

	if ((m->counter % SAMPLES_PER_PIXEL) == 0) {
		m->picf = 256 * (m->picf / SAMPLES_PER_PIXEL - 1000) / trx->bandwidth;

		trx_put_rx_picdata(CLAMP(m->picf, 0.0, 255.0));

		m->picf = 0.0;
	}
}
*/

static void recvchar(Trx *trx, int c)
{
	Mfsk *m = (Mfsk*) trx->modem;
//	int h, w;
//	BOOL color;

	if (c == -1)
		return;

/*
	memmove(m->picheader, m->picheader + 1, sizeof(m->picheader) - 1);

	m->picheader[sizeof(m->picheader) - 2] = (c == 0) ? ' ' : c;
	m->picheader[sizeof(m->picheader) - 1] = 0;

	if (picture_check_header(m->picheader, &w, &h, &color)) {
		if (m->symbolbit == 4)
			m->rxstate = RX_STATE_PICTURE_START_1;
		else
			m->rxstate = RX_STATE_PICTURE_START_2;
		
		m->picturesize = SAMPLES_PER_PIXEL * w * h * (color ? 3 : 1);
		m->counter = 0;

		if (color)
			trx_put_rx_picdata(('C' << 24) | (w << 12) | h);
		else
			trx_put_rx_picdata(('M' << 24) | (w << 12) | h);

		memset(m->picheader, ' ', sizeof(m->picheader));
	}
*/

	trx_put_rx_char(trx, c);
}

static void recvbit(Trx *trx, int bit)
{
	Mfsk *m = (Mfsk*)trx->modem;
	int			 c;

	m->datashreg <<= 1;
	if (bit)
		m->datashreg |= 1;

	/* search for "001" */
	if ((m->datashreg & 7) == 1) {
		/* the "1" belongs to the next symbol */
		c = varidec(m->datashreg >> 1);

		recvchar(trx, c);

		/* we already received this */
		m->datashreg = 1;
	}
}

static void decodesymbol(Trx *trx, unsigned char symbol)
{
	Mfsk *m = (Mfsk*)trx->modem;
	int			 c, met;

	m->symbolpair[0] = m->symbolpair[1];
	m->symbolpair[1] = symbol;
	m->symcounter	 = m->symcounter ? 0 : 1;

	/* MFSK16 doesn't need a vote */
	if (trx->mode == MODE_MFSK16 && m->symcounter)
		return;

	if (m->symcounter) {
		if ((c = viterbi_decode(m->dec1, m->symbolpair, &met)) == -1)
			return;
		m->met1 -= m->met1 >> 5;
		m->met1 += met;
		if (m->met1 < m->met2)
			return;
		m->metric = (m->met1 + 40) / 80;
	} else {
		if ((c = viterbi_decode(m->dec2, m->symbolpair, &met)) == -1)
			return;
		m->met2 -= m->met2 >> 5;
		m->met2 += met;
		if (m->met2 < m->met1)
			return;
		m->metric = (m->met2 + 40) / 80;
	}

	if (trx->squelchon && m->metric < trx->iSquelchThreshold)
		return;

	recvbit(trx, c);
/*
	{
		char bufout[256];
		sprintf(bufout, "met1=%d met2=%d, metric=%d%% %d\r\n", 
			m->met1, m->met2, m->metric, m->met1 > m->met2 ? 1 : 2);
		OutputDebugString(bufout);
	}
*/
}

static void softdecode(Trx *trx, complex *bins)
{
	Mfsk		   *m = (Mfsk*)trx->modem;
	int				tone;
	int     		sum = 0;
	int				b[5];		// 4 or 5 bits
	unsigned char   symbols[5]; // 4 or 5 bits
	int				i, j, k;
	complex		   *z;

	memset(b, 0, m->symbits * sizeof(int));

	/* gray decode and form soft decision samples */
	for (i = 0; i < m->numtones; ++ i) {
		j = graydecode(i);
		k = trx->reverse ? ((m->numtones - 1) - i) : i;
		z = bins + k;
		_ASSERT(fabs(z->re) < 65536 && fabs(z->im) < 65536);
		tone = ihypot(z->re, z->im);
		tone >>= m->symbits;
		for (k = 0; k < m->symbits; ++ k)
			b[k] += (j & (1 << (m->symbits - k - 1))) ? tone : - tone;
		sum += tone;
	}
	_ASSERT(sum < 65536);

	/* shift to range 0...255 */
	if (sum == 0) {
		for (i = 0; i < m->symbits; ++ i)
			symbols[i] = 128;
	} else {
		for (i = 0; i < m->symbits; ++ i) {
			_ASSERT(fabs(b[i]) < 32768);
			j = 128 + (( (b[i] << 16) / sum + (1 << 8) ) >> 9);
			if (j < 0)
				j = 0;
			else if (j > 255)
				j = 255;
			symbols[i] = j;
		}
	}

	interleave_syms(m->rxinlv, symbols);

	for (i = 0; i < m->symbits; ++ i) {
		// The picture decoder needs to know exactly when the
		// header is detected.
		m->symbolbit = i + 1;
		decodesymbol(trx, symbols[i]);
	}
}

/// Find bucket with the highest energy
static int harddecode(Trx *trx, complex *inp)
{
	Mfsk *m		= (Mfsk*)trx->modem;
	unsigned	 x;
	int			 i;
	unsigned     max	= 0;
	int			 symbol	= 0;
	complex		*in		= inp;

	for (i = 0; i < m->numtones; ++ i, ++ in) {
		_ASSERT(fabs(in->re) < 65536);
		_ASSERT(fabs(in->im) < 65536);
		x = in->re * in->re + in->im * in->im;
		if (x > max) {
			max		= x;
			symbol	= i;
		}
	}

	return symbol;
}

/*
static void update_syncscope(Mfsk *m)
{
	int data[2048];
	int i, j, len;
	complex *z;

	len = 100;
//	len = 2 * m->symlen;
	for (i = 0; i < len; ++ i) {
		j = (i + m->pipeptr) % (2 * m->symlen);
		z = m->pipe[j].vector + m->prev1symbol;
		data[i] = ihypot(z->re, z->im);
	}

	trx_set_scope(data, len, TRUE);
}
*/

static void synchronize(Mfsk *m)
{
	int				i;
	unsigned		max = 0;
	int				syn = -1;
	int				nSamples	= m->symlen >> MFSK_UNDERSAMPLING;
	int				iPipeStart	= m->pipeptr + (nSamples >> 1);

	if (iPipeStart >= nSamples * 3)
		iPipeStart -= nSamples * 3;

	for (i = (nSamples >> 1); i <  nSamples + (nSamples >> 1); ++ i) {
//	for (i = 0; i < nSamples << 1; ++ i) {
		int			iPipe	= ++ iPipeStart;
		complex	   *c		= m->dftcoeffs + m->prev1symbol * nSamples;
		complex		sum;
		int			iSample;
		unsigned	val;

		if (iPipe == nSamples * 3) {
			iPipe	   = 0;
			iPipeStart = 0;
		}

		sum.re = 0;
		sum.im = 0;
		for (iSample = 0; iSample < nSamples; ++ iSample, ++ c) {
			complex *z = m->pipe + iPipe;
			sum.re += z->re * c->re - z->im * c->im;
			sum.im += z->im * c->re + z->re * c->im;
			if (++ iPipe == nSamples * 3)
				iPipe = 0;
		}
		sum.re = (sum.re + 0x7fff) >> 16;
		sum.im = (sum.im + 0x7fff) >> 16;

		val = sum.re * sum.re + sum.im * sum.im;
		if (val > max) {
			max = val;
			syn = i;
		}
	}

	m->synccounter += (int)floor(((syn - nSamples) << MFSK_UNDERSAMPLING) / 16.0 + 0.5);
/*
	{
		char bufout[256];
		sprintf(bufout, "symbol: %d, sync: %d\n", m->prev1symbol, syn);
		OutputDebugString(bufout);
	}
*/
}

static void afc(Trx *trx)
{
	Mfsk *m = (Mfsk*)trx->modem;
	int   diffphasex, diffphasey, phase;
	int   err;

	if (! trx->afcon || m->currsymbol != m->prev1symbol || m->metric < trx->iSquelchThreshold)
		return;

	diffphasex = m->prev1vector.re * m->currvector.re + m->prev1vector.im * m->currvector.im;
	diffphasey = m->prev1vector.re * m->currvector.im - m->prev1vector.im * m->currvector.re;
	// phase is a 16 bit number from -32768 to +32768
	phase      = iatan2(diffphasey, diffphasex);
	err		   = (int)(((double)phase * SampleRate / 32768.0) / m->symlen);
//	err		   = (phase * SampleRate) >> (m->symbits + 16 - FREQ_FRAC_BITS - 1);

	if (err > - m->tonespacing &&  err < m->tonespacing)
//		trx->frequency += (err + 1) >> 2;
		trx->frequency += (err + 3) >> 3;
//		trx->frequency += (err + 7) >> 4;
}

static complex* do_dft(Mfsk *s)
{
	int				iTone;
	int				iSample;
	int				nSamples	= s->symlen >> MFSK_UNDERSAMPLING;
	int				iPipeStart	= s->pipeptr - nSamples + 1;
	complex		   *c			= s->dftcoeffs;

	if (iPipeStart < 0)
		iPipeStart += nSamples * 3;

	for (iTone = 0; iTone < s->numtones; ++ iTone) {
		complex *sum   = s->aTones + iTone;
		int		 iPipe = iPipeStart;
		sum->re = 0;
		sum->im = 0;
		for (iSample = 0; iSample < nSamples; ++ iSample, ++ c) {
			complex *z = s->pipe + iPipe;
			sum->re += z->re * c->re - z->im * c->im;
			sum->im += z->im * c->re + z->re * c->im;
			if (++ iPipe == nSamples * 3)
				iPipe = 0;
		}
		sum->re = (sum->re + 0x7fff) >> 16;
		sum->im = (sum->im + 0x7fff) >> 16;
	}

	return s->aTones;
}

int mfsk_rxprocess(Trx *trx, short *buf, int len)
{
	Mfsk *m = (Mfsk*)trx->modem;
	complex z;
	int i;

	// Basetone is always 15.625 Hz
	unsigned freqdelta = - m->basetone * 250;

	while (len-- > 0) {
// bits of trx->frequency = 12 /* 0-4kHz */ + FREQ_FRAC_BITS
//		unsigned iPhaseDelta2 = (unsigned)((65536.0 * 65536.0) * trx->frequency / (SampleRate << FREQ_FRAC_BITS));
//		unsigned iPhaseDelta = 
//			(((trx->frequency << (32 - (12 + FREQ_FRAC_BITS))) / SampleRate) << 12) + (1 << 11);
		unsigned iPhaseDelta = 33554 /* 2^32  / (SampleRate << FREQ_FRAC_BITS) = 3354.432 */ * (trx->frequency + freqdelta);

		// Mix with the internal NCO, create analytic signal
		mixer_mix(*(buf ++), m->phaseacc, &z);
		m->phaseacc += iPhaseDelta;
		z.re = (z.re + 0x3fff) >> 15;
		z.im = (z.im + 0x3fff) >> 15;

//		dbgwaterfall_set_data(&s_DbgWaterfall, &z, 1);

		if (! filter_run(m->filt, z, &z))
			continue;
//		z.re = (z.re + 0x3fff) >> 15;
//		z.im = (z.im + 0x3fff) >> 15;

//		z.re = (z.re + 0x1fff) >> 14;
//		z.im = (z.im + 0x1fff) >> 14;

//		z.re = (z.re + 0x0fff) >> 13;
//		z.im = (z.im + 0x0fff) >> 13;

		z.re = (z.re + 0x07ff) >> 12;
		z.im = (z.im + 0x07ff) >> 12;

		if (z.re > 0x3fff)
			z.re = 0x3fff;
		else if (z.re < -0x3fff)
			z.re = -0x3fff;
		if (z.im > 0x3fff)
			z.im = 0x3fff;
		else if (z.im < -0x3fff)
			z.im = -0x3fff;

		m->pipe[m->pipeptr] = z;

//		dbgwaterfall_set_data(&s_DbgWaterfall, &z, 1);

/*
		if (m->rxstate == RX_STATE_PICTURE_START_2) {
			if (m->counter++ == 352) {
				m->counter = 0;
				m->rxstate = RX_STATE_PICTURE;
			}
			continue;
		}
		if (m->rxstate == RX_STATE_PICTURE_START_1) {
			if (m->counter++ == 352 + m->symlen) {
				m->counter = 0;
				m->rxstate = RX_STATE_PICTURE;
			}
			continue;
		}

		if (m->rxstate == RX_STATE_PICTURE) {
			if (m->counter++ == m->picturesize) {
				m->counter = 0;
				m->rxstate = RX_STATE_DATA;
			} else
				recvpic(trx, z);
			continue;
		}
*/

		m->synccounter -= (1 << MFSK_UNDERSAMPLING);
		if (m->synccounter <= 0) {
			complex *aBins = do_dft(m);

			{
				int aAmps[32];
				for (i = 0; i < m->numtones; ++ i)
					aAmps[i] = ihypot(aBins[i].re, aBins[i].im);
				trx_set_scope(aAmps, m->numtones, TRUE);
			}

			m->synccounter += m->symlen;

			// Find bucket with the highest energy
			m->currsymbol = harddecode(trx, aBins);
			m->currvector = aBins[m->currsymbol];

			/* decode symbol */
			softdecode(trx, aBins);

			/* update the scope */
//			update_syncscope(m);

			/* symbol sync */
			if (m->currsymbol != m->prev1symbol && m->prev1symbol != m->prev2symbol)
				synchronize(m);

			/* frequency tracking */
			afc(trx);

			m->prev2symbol = m->prev1symbol;
			m->prev1symbol = m->currsymbol;
			m->prev1vector = m->currvector;
		}

		if (++ m->pipeptr >= (int)((m->symlen * 3) >> MFSK_UNDERSAMPLING))
			m->pipeptr = 0;
	}

	return 0;
}
