/*
 * Waterfall Spectrum Analyzer Widget
 * Copyright (C) 2001,2002,2003 Tomi Manninen <oh2bns@sral.fi>
 *
 * Based on the Spectrum Widget by Thomas Sailer <sailer@ife.ee.ethz.ch>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include "waterfall.h"
#include "ColorTable.h"
#include <memory.h>
#include <stdio.h>
#include <tchar.h>

#include "../misc/int_fft.h"

#include "../Resource.h"

#define	RULER_HEIGHT	14

/// Forward reference to the waterfall window procedure
LRESULT CALLBACK waterfall_wndproc(HWND, UINT, WPARAM, LPARAM);

const TBYTE WIN32CLASS_WATERFALL[] = _T("WATERFALL");

/// Register class of the waterfall window
ATOM waterfall_register_class(HINSTANCE hInstance)
{
	WNDCLASS	wc;

	wc.style			= CS_HREDRAW | CS_VREDRAW;
	wc.lpfnWndProc		= (WNDPROC) waterfall_wndproc;
	wc.cbClsExtra		= 0;
	wc.cbWndExtra		= 0;
	wc.hInstance		= hInstance;
	wc.hIcon			= 0;
	wc.hCursor			= 0;
	wc.hbrBackground	= (HBRUSH) GetStockObject(WHITE_BRUSH);
	wc.lpszMenuName		= 0;
	wc.lpszClassName	= WIN32CLASS_WATERFALL;

	return RegisterClass(&wc);
}

Waterfall* waterfall_new()
{
	Waterfall *wf = (Waterfall*)malloc(sizeof(Waterfall));
	if (wf == 0)
		return 0;
	if (! waterfall_init(wf)) {
		waterfall_destroy(wf);
		return 0;
	}
	return wf;
}

void waterfall_free(Waterfall *wf)
{
	if (wf == 0)
		return;
	waterfall_destroy(wf);
	free(wf);
}

static BOOL alloc_pixbuf(Waterfall *wf, BOOL restart)
{
	RECT	rect;
	int		nWidth, nHeight;
	int		nSize;

	if (! wf->hWnd)
		return FALSE;

	GetClientRect(wf->hWnd, &rect);
	nWidth  = rect.right - rect.left;
	nHeight = rect.bottom - rect.top - RULER_HEIGHT;
	nSize   = (nHeight * wf->fftlen) >> 1;

	if (restart) {
		if (nSize != wf->pixbufsize)
			wf->pixbuf = (COLORREF*)realloc((void*)wf->pixbuf, nSize * sizeof(COLORREF));
		wf->pixbufptr  = 0;
		wf->pixbufsize = nSize;
		if (wf->pixbuf == 0)
			return FALSE;
		memset(wf->pixbuf, 0, nSize * sizeof(COLORREF));
	} else if (nSize != wf->pixbufsize) {
		COLORREF *pBuf = (COLORREF*)malloc(nSize * sizeof(COLORREF));
		int       nPtr = 0;
		int		  nCopyLen = (wf->pixbufsize > nSize) ? nSize : wf->pixbufsize; // min
		int		  nCopyFirst;
		if (pBuf == 0)
			return FALSE;
		nPtr = wf->pixbufptr - nCopyLen;
		if (nPtr < 0)
			nPtr += wf->pixbufsize;
		nCopyFirst = wf->pixbufsize - nPtr;
		if (nCopyFirst >= nCopyLen) {
			memcpy(pBuf, wf->pixbuf + nPtr, nCopyLen * sizeof(COLORREF));
		} else {
			memcpy(pBuf, wf->pixbuf + nPtr, nCopyFirst * sizeof(COLORREF));
			memcpy(pBuf + nCopyFirst, wf->pixbuf, (nCopyLen - nCopyFirst) * sizeof(COLORREF));
		}
		if (nCopyLen < nSize)
			memset(pBuf + nCopyLen, 0, (nSize - nCopyLen) * sizeof(COLORREF));
		free(wf->pixbuf);
		wf->pixbuf	   = pBuf;
		wf->pixbufptr  = (nCopyLen == nSize) ? 0 : nCopyLen;
		wf->pixbufsize = nSize;
	}

	{
		BITMAPINFO  bmi;
		DeleteObject(wf->hBitmap);
		memset(&bmi, 0, sizeof(BITMAPINFO));
		nWidth = ((nWidth + 3) >> 2) << 2;
		//FIXME
		nHeight = 1;
		bmi.bmiHeader.biSize		= sizeof(bmi.bmiHeader);
		bmi.bmiHeader.biWidth		= nWidth;
		bmi.bmiHeader.biHeight		= nHeight;
		bmi.bmiHeader.biSizeImage   = nWidth * nHeight * 3; 
		bmi.bmiHeader.biPlanes		= 1;
		bmi.bmiHeader.biBitCount	= 24;
		bmi.bmiHeader.biCompression = BI_RGB;
		bmi.bmiHeader.biXPelsPerMeter = 3780;
		bmi.bmiHeader.biYPelsPerMeter = 3780;
		{
			HDC hDC = GetDC(NULL);
			wf->hBitmap = CreateDIBSection(hDC, &bmi, DIB_RGB_COLORS, (void**)&wf->pBitmapBits, NULL, 0);
			//FIXME
			memset(wf->pBitmapBits, 0, bmi.bmiHeader.biSizeImage);
			ReleaseDC(NULL, hDC);
		}
	}

	if ((wf->fftlen >> 1) - wf->ruler_ref_x < nWidth)
		wf->ruler_ref_x = (wf->fftlen >> 1) - nWidth;
	if (wf->ruler_ref_x < 0)
		wf->ruler_ref_x = 0;

	return TRUE;
}

BOOL waterfall_init(Waterfall *wf)
{
	memset(wf, 0, sizeof(Waterfall));

	if (FALSE) {
		wf->nColorTableLen = 91;
		wf->aColorTable = malloc(wf->nColorTableLen * sizeof(COLORREF));
		CreatePowerColorGradient16bit(wf->aColorTable, wf->nColorTableLen);
	} else {
		wf->nColorTableLen = 6 * 31 + 1;
		wf->aColorTable = malloc(wf->nColorTableLen * sizeof(COLORREF));
		CreatePowerColorGradientSimple(5, wf->aColorTable, wf->nColorTableLen);
	}

	wf->pointer = -1;

	waterfall_load(wf);

	waterfall_set_magnification(wf, wf->config.magnification, FALSE);
//	wf->fftlen					= WATERFALL_FFTLEN_1;
	wf->samplerate				= 8000.0;

	wf->frequency				= 1000.0;
	wf->carrierfreq				= 0.0;
	wf->bw						= 200.0;

//	calculate_frequencies(wf);

	wf->lsb						= FALSE;

	wf->inbuf	= malloc(WATERFALL_FFTLEN_MAX * sizeof(short));
	wf->specbuf = malloc(WATERFALL_FFTLEN_MAX * sizeof(short));
	wf->peakbuf = malloc(WATERFALL_FFTLEN_MAX * sizeof(short));
	wf->loudbuf = (UINT*)malloc(WATERFALL_FFTLEN_MAX * sizeof(UINT));
	memset(wf->inbuf,   0, WATERFALL_FFTLEN_MAX * sizeof(short));
	memset(wf->specbuf, 0, WATERFALL_FFTLEN_MAX * sizeof(short));
	memset(wf->peakbuf, 0, WATERFALL_FFTLEN_MAX * sizeof(short));
	memset(wf->loudbuf, 0, WATERFALL_FFTLEN_MAX * sizeof(UINT));

	wf->iAverage  = 0;
//	wf->nAverages = 32;
	wf->nAverages = 1;

//    wf->kiss_fftr_state = kiss_fftr_alloc(wf->fftlen, 0, 0, 0);

	return TRUE;
}

BOOL waterfall_create(Waterfall *wf, DWORD dwStyle,	
	int x, int y, int nWidth, int nHeight, HWND hWndParent, HANDLE hInstance)
{
	wf->hWnd = CreateWindow(WIN32CLASS_WATERFALL, NULL, dwStyle,
		x, y, nWidth, nHeight, hWndParent, NULL, hInstance, (void*)wf);
	if (! wf->hWnd)
		return FALSE;
	SetWindowLong(wf->hWnd, GWL_USERDATA, (long)wf);
	{
		LOGFONT logfont;
		memset(&logfont, 0, sizeof(LOGFONT));
		logfont.lfHeight  = -9;
		logfont.lfWeight  = FW_NORMAL;
		logfont.lfCharSet = ANSI_CHARSET;
		_tcscpy(logfont.lfFaceName, _T("Tahoma"));
		wf->hFont = CreateFontIndirect(&logfont);
	}

//	calculate_frequencies(wf);
	return alloc_pixbuf(wf, FALSE);
}

void waterfall_destroy(Waterfall *wf)
{
	DeleteObject(wf->hBitmap);
	DeleteObject(wf->hFont);
	DestroyWindow(wf->hWnd);
	wf->hWnd		= NULL;

	free(wf->aColorTable);
	wf->aColorTable		= 0;
	wf->nColorTableLen	= 0;

	free(wf->pixbuf);
	wf->pixbuf		= NULL;
	wf->pixbufptr	= 0;
	wf->pixbufsize	= 0;

	free(wf->inbuf);
	free(wf->specbuf);
	free(wf->peakbuf);
	wf->inbuf		= NULL;
	wf->specbuf		= NULL;
	wf->peakbuf		= NULL;

//	kiss_fftr_free(wf->kiss_fftr_state);
//	wf->kiss_fftr_state = NULL;
}

void waterfall_load(Waterfall *wf)
{
	HKEY  hKeyStnInfo = 0;
	DWORD dwType	  = REG_DWORD;
	DWORD dwSize      = 4;

	waterfall_load_defaults(wf);
	if (RegOpenKeyEx(HKEY_CURRENT_USER, _T("Software\\OK1IAK\\PocketDigi\\Waterfall"), 0,
		KEY_ENUMERATE_SUB_KEYS | KEY_QUERY_VALUE | KEY_READ, &hKeyStnInfo) != ERROR_SUCCESS)
		return;

	dwType = REG_DWORD; dwSize = 4;
	RegQueryValueEx(hKeyStnInfo, _T("Magnification"), 0, &dwType, 
		(LPBYTE)&wf->config.magnification, &dwSize);

	dwType = REG_DWORD; dwSize = 4;
	RegQueryValueEx(hKeyStnInfo, _T("Speed"), 0, &dwType, 
		(LPBYTE)&wf->config.speed, &dwSize);

	dwType = REG_DWORD; dwSize = 4;
	RegQueryValueEx(hKeyStnInfo, _T("Threshold"), 0, &dwType, 
		(LPBYTE)&wf->config.iDecibelsThreshold, &dwSize);

	dwType = REG_DWORD; dwSize = 4;
	RegQueryValueEx(hKeyStnInfo, _T("Range"), 0, &dwType, 
		(LPBYTE)&wf->config.iDecibelsRange, &dwSize);

	RegCloseKey(hKeyStnInfo);
}

void waterfall_load_defaults(Waterfall *wf)
{
	wf->config.magnification		= WATERFALL_MAG_1;
	wf->config.mode					= WATERFALL_MODE_NORMAL;
	wf->config.speed				= WATERFALL_SPEED_NORMAL;
	wf->config.direction			= TRUE;
	wf->config.iDecibelsThreshold	= 10;
	wf->config.iDecibelsRange		= 75;
}

void waterfall_store(Waterfall *wf)
{
	HKEY  hKeyStnInfo   = 0;
	DWORD dwDisposition = 0;
	DWORD dwType	    = REG_SZ;
	DWORD dwSize        = 0;

	if (RegCreateKeyEx(HKEY_CURRENT_USER, _T("Software\\OK1IAK\\PocketDigi\\Waterfall"), 0, 0,
		REG_OPTION_NON_VOLATILE, KEY_ALL_ACCESS, 0, &hKeyStnInfo, &dwDisposition) != ERROR_SUCCESS)
		return;

	RegSetValueEx(hKeyStnInfo, _T("Magnification"), 0, REG_DWORD,
		(LPBYTE)&wf->config.magnification, 4);
	RegSetValueEx(hKeyStnInfo, _T("Speed"), 0, REG_DWORD,
		(LPBYTE)&wf->config.speed, 4);
	RegSetValueEx(hKeyStnInfo, _T("Threshold"), 0, REG_DWORD,
		(LPBYTE)&wf->config.iDecibelsThreshold, 4);
	RegSetValueEx(hKeyStnInfo, _T("Range"), 0, REG_DWORD,
		(LPBYTE)&wf->config.iDecibelsRange, 4);
}

void waterfall_set_magnification(Waterfall *wf, wf_mag_t mag, BOOL bUpdateUI)
{
	switch (mag) {
	case WATERFALL_MAG_0:
		if (wf->fftlen == WATERFALL_FFTLEN_0)
			return;
		wf->fftlen = WATERFALL_FFTLEN_0;
		break;
	case WATERFALL_MAG_1:
		if (wf->fftlen == WATERFALL_FFTLEN_1)
			return;
		wf->fftlen = WATERFALL_FFTLEN_1;
		break;
	case WATERFALL_MAG_2:
		if (wf->fftlen == WATERFALL_FFTLEN_2)
			return;
		wf->fftlen = WATERFALL_FFTLEN_2;
		break;
	case WATERFALL_MAG_4:
		if (wf->fftlen == WATERFALL_FFTLEN_4)
			return;
		wf->fftlen = WATERFALL_FFTLEN_4;
		break;
	default:
//		g_warning("Unsupported magnification: %d\n", mag);
		return;
	}

	wf->config.magnification = mag;

	if (! bUpdateUI)
		return;

	alloc_pixbuf(wf, TRUE);

	memset(wf->specbuf, 0, sizeof(short) * WATERFALL_FFTLEN_MAX);
	memset(wf->peakbuf, 0, sizeof(short) * WATERFALL_FFTLEN_MAX);
	memset(wf->inbuf,   0, sizeof(short) * WATERFALL_FFTLEN_MAX);
	wf->inptr = 0;

//FIXME
//	calculate_frequencies(wf);

	InvalidateRect(wf->hWnd, 0, FALSE);
}

#if 0
static void waterfall_size_allocate(GtkWidget *widget,
				    GtkAllocation *allocation)
{
	Waterfall *wf;
	gint width, height;

	g_return_if_fail(widget != NULL);
	g_return_if_fail(IS_WATERFALL(widget));
	g_return_if_fail(allocation != NULL);

	wf = WATERFALL(widget);

	g_mutex_lock(wf->mutex);

	width = MIN(allocation->width, wf->fftlen / 2);
	height = allocation->height;


	widget->allocation = *allocation;
	widget->allocation.width = width;
	widget->allocation.height = height;

	wf->inptr = 0;

	alloc_pixbuf(wf, FALSE);

	calculate_frequencies(wf);

	if (GTK_WIDGET_REALIZED(widget)) {
		if (wf->pixmap)
			gdk_pixmap_unref(wf->pixmap);

		wf->pixmap = gdk_pixmap_new(widget->window, width, height, -1);

		gdk_window_move_resize(widget->window,
				       allocation->x, allocation->y,
				       allocation->width, allocation->height);

		waterfall_send_configure(WATERFALL(widget));
	}

	g_mutex_unlock(wf->mutex);
}

static gint waterfall_button_press(GtkWidget *widget, GdkEventButton *event)
{
	Waterfall *wf = WATERFALL(widget);
	gdouble resolution, span, f;

	if (wf->config.mode == WATERFALL_MODE_SCOPE)
		return FALSE;

	if (event->button != 1)
		return FALSE;

	if (event->y < RULER_HEIGHT) {
		gtk_grab_add(widget);

		wf->ruler_drag = TRUE;
		wf->ruler_ref_x = event->x;
		wf->ruler_ref_f = wf->centerfreq;

		return FALSE;
	}

	if (wf->fixed)
		return FALSE;

	resolution = wf->samplerate / wf->fftlen;
	span = resolution * widget->allocation.width;

	f = wf->centerfreq - span / 2 + event->x * resolution;

	f = MAX(f, wf->bw / 2.0);
	f = MIN(f, wf->samplerate / 2.0 - wf->bw / 2.0);

	wf->frequency = f;

	calculate_frequencies(wf);

	gtk_signal_emit(GTK_OBJECT(wf), 
			waterfall_signals[FREQUENCY_SET_SIGNAL],
			wf->frequency);

	return FALSE;
}

static gint waterfall_button_release(GtkWidget *widget, GdkEventButton *event)
{
	Waterfall *wf = WATERFALL(widget);

	if (wf->ruler_drag) {
		wf->ruler_drag = FALSE;
		gtk_grab_remove(widget);
	}

	return FALSE;
}

static gint waterfall_motion_notify(GtkWidget *widget, GdkEventMotion *event)
{
	Waterfall *wf = WATERFALL(widget);
	GdkModifierType mods;
	gint x, y;

	if (wf->config.mode == WATERFALL_MODE_SCOPE)
		return FALSE;

	if (event->is_hint || (event->window != widget->window)) {
		gdk_window_get_pointer(widget->window, &x, &y, &mods);
	} else {
		x = event->x;
		y = event->y;
		mods = event->state;
	}

	if (wf->ruler_drag && (mods & GDK_BUTTON1_MASK)) {
		gdouble res = wf->samplerate / wf->fftlen;

		wf->centerfreq = wf->ruler_ref_f + res * (wf->ruler_ref_x - x);

		calculate_frequencies(wf);
		return FALSE;
	}

	if (y < RULER_HEIGHT) {
		wf->pointer = -1;

		if (!wf->ruler_cursor_set) {
			gdk_window_set_cursor(widget->window, wf->ruler_cursor);
			wf->ruler_cursor_set = TRUE;
		}
	} else {
		if (wf->fixed)
			wf->pointer = -1;
		else
			wf->pointer = x;

		if (wf->ruler_cursor_set) {
			gdk_window_set_cursor(widget->window, NULL);
			wf->ruler_cursor_set = FALSE;
		}
	}

	set_idle_callback(wf);

	return FALSE;
}

void waterfall_set_window(Waterfall *wf, wf_window_t type)
{
	g_return_if_fail(wf != NULL);
	g_return_if_fail(IS_WATERFALL(wf));

	g_mutex_lock(wf->mutex);
	setwindow(wf->fft_window, wf->fftlen, type);
	g_mutex_unlock(wf->mutex);

	wf->config.window = type;
}

void calculate_frequencies(Waterfall *wf)
{
	gdouble start, stop, span, res;
	gint width;

	width = GTK_WIDGET(wf)->allocation.width;
	res   = wf->samplerate / wf->fftlen;
	span  = res * width;

	start = wf->centerfreq - span / 2;

	if (start > (wf->frequency - wf->bw / 2))
		start = wf->frequency - wf->bw / 2;

	if (start < 0)
		start = 0;

	stop = start + span;

	if (stop < (wf->frequency + wf->bw / 2))
		stop = wf->frequency + wf->bw / 2;

	if (stop > wf->samplerate / 2)
		stop = wf->samplerate / 2;

	wf->centerfreq = stop - span / 2;
	wf->startfreq = stop - span;
	wf->resolution = res;
}

void waterfall_set_frequency(Waterfall *wf, gdouble f)
{
	g_return_if_fail(wf != NULL);
	g_return_if_fail(IS_WATERFALL(wf));

	f = MAX(f, wf->bw / 2.0);
	f = MIN(f, wf->samplerate / 2.0 - wf->bw / 2.0);

	wf->frequency = f;

	calculate_frequencies(wf);

	gtk_signal_emit(GTK_OBJECT(wf),
			waterfall_signals[FREQUENCY_SET_SIGNAL],
			wf->frequency);

	set_idle_callback(wf);
}

void waterfall_set_center_frequency(Waterfall *wf, gdouble f)
{
	g_return_if_fail(wf != NULL);
	g_return_if_fail(IS_WATERFALL(wf));

	wf->centerfreq = f;

	calculate_frequencies(wf);

	set_idle_callback(wf);
}

void waterfall_set_carrier_frequency(Waterfall *wf, gdouble f)
{
	g_return_if_fail(wf != NULL);
	g_return_if_fail(IS_WATERFALL(wf));

	wf->carrierfreq = f;

	set_idle_callback(wf);
}

void waterfall_set_samplerate(Waterfall *wf, gint samplerate)
{
	g_return_if_fail(wf != NULL);
	g_return_if_fail(IS_WATERFALL(wf));

	g_mutex_lock(wf->mutex);

	if (wf->samplerate != samplerate) {
		memset(wf->pixbuf, 0, wf->pixbufsize * sizeof(COLORREF));
		wf->samplerate = samplerate;
	}

	g_mutex_unlock(wf->mutex);

	set_idle_callback(wf);
}

void waterfall_set_bandwidth(Waterfall *wf, gdouble bw)
{
	g_return_if_fail(wf != NULL);
	g_return_if_fail(IS_WATERFALL(wf));

	if (wf->bw != bw) {
		wf->bw = bw;
		set_idle_callback(wf);
	}
}

void waterfall_set_centerline(Waterfall *wf, gboolean flag)
{
	g_return_if_fail(wf != NULL);
	g_return_if_fail(IS_WATERFALL(wf));

	wf->centerline = flag;

	set_idle_callback(wf);
}

void waterfall_set_mode(Waterfall *wf, wf_mode_t mode)
{
	g_return_if_fail(wf != NULL);
	g_return_if_fail(IS_WATERFALL(wf));

	wf->config.mode = mode;

	set_idle_callback(wf);
}

void waterfall_set_dir(Waterfall *wf, gboolean dir)
{
	g_return_if_fail(wf != NULL);
	g_return_if_fail(IS_WATERFALL(wf));

	wf->config.direction = dir;
}

void waterfall_set_pause(Waterfall *wf, gboolean flag)
{
	g_return_if_fail(wf != NULL);
	g_return_if_fail(IS_WATERFALL(wf));

	wf->paused = flag;
}

void waterfall_set_fixed(Waterfall *wf, gboolean flag)
{
	g_return_if_fail(wf != NULL);
	g_return_if_fail(IS_WATERFALL(wf));

	wf->fixed = flag;
}

void waterfall_set_lsb(Waterfall *wf, gboolean flag)
{
	g_return_if_fail(wf != NULL);
	g_return_if_fail(IS_WATERFALL(wf));

	wf->lsb = flag;

	set_idle_callback(wf);
}

void waterfall_get_config(Waterfall *wf, wf_config_t *config)
{
	g_return_if_fail(wf != NULL);
	g_return_if_fail(IS_WATERFALL(wf));
	g_return_if_fail(config != NULL);

	*config = wf->config;
}

#endif /* 0 */

/* ---------------------------------------------------------------------- */

/*
static inline gint ftox(Waterfall *wf, gdouble f)
{
	return (gint) ((f - wf->startfreq) / wf->resolution + 0.5);
}

static inline gint ftow(Waterfall *wf, gdouble f)
{
	return (gint) (f / wf->resolution + 0.5);
}

static inline gdouble xtof(Waterfall *wf, gint x)
{
	return wf->startfreq + x * wf->resolution;
}

static void draw_markers(Waterfall *wf)
{
	gint x1, x2, x3;
	GdkGC *gc;

	x1 = ftox(wf, wf->frequency - (wf->bw / 2.0));
	x2 = ftox(wf, wf->frequency);
	x3 = ftox(wf, wf->frequency + (wf->bw / 2.0));

	gc = (wf->fixed) ? wf->pointer3_gc : wf->pointer2_gc;

	gdk_draw_line(wf->pixmap, gc,
		      x1, RULER_HEIGHT,
		      x1, GTK_WIDGET(wf)->allocation.height - 1);
	if (wf->centerline) {
		gdk_draw_line(wf->pixmap, gc,
			      x2, RULER_HEIGHT,
			      x2, GTK_WIDGET(wf)->allocation.height - 1);
	}
	gdk_draw_line(wf->pixmap, gc,
		      x3, RULER_HEIGHT,
		      x3, GTK_WIDGET(wf)->allocation.height - 1);

	if (wf->fixed || wf->pointer == -1)
		return;

	x1 = ftox(wf, xtof(wf, wf->pointer) - wf->bw / 2);
	x2 = wf->pointer;
	x3 = ftox(wf, xtof(wf, wf->pointer) + wf->bw / 2);

	gc = wf->pointer1_gc;

	gdk_draw_line(wf->pixmap, gc,
		      x1, RULER_HEIGHT,
		      x1, GTK_WIDGET(wf)->allocation.height - 1);
	if (wf->centerline) {
		gdk_draw_line(wf->pixmap, gc,
			      x2, RULER_HEIGHT,
			      x2, GTK_WIDGET(wf)->allocation.height - 1);
	}
	gdk_draw_line(wf->pixmap, gc,
		      x3, RULER_HEIGHT,
		      x3, GTK_WIDGET(wf)->allocation.height - 1);
}

struct tic {
	gboolean major;
	gdouble freq;
	gchar str[32];
	gint strw;
	gint strh;
	gint strx;
	gint x;

	struct tic *next;
};

static struct tic *build_tics(Waterfall *wf)
{
	struct tic *list, *p;
	gdouble realfreq, f;
	gint i, freq, width;

	list = NULL;

	width = GTK_WIDGET(wf)->allocation.width;
	f = wf->startfreq;

	for (i = 0; i < width; i++) {
		if (wf->lsb)
			realfreq = f - wf->carrierfreq;
		else
			realfreq = f + wf->carrierfreq;

		realfreq = fabs(realfreq);

		freq = 100 * floor(realfreq / 100.0 + 0.5);

		if (freq < realfreq || freq >= realfreq + wf->resolution) {
			f += wf->resolution;
			continue;
		}

		p = g_new0(struct tic, 1);

		p->major = FALSE;
		p->freq = freq;
		p->x = i;

		if ((freq % 500) == 0) {
			GdkFont *font;
			gint hz, khz;

			khz = freq / 1000;
			hz  = freq % 1000;

			if (khz > 9)
				sprintf(p->str, "%d.%03d", khz, hz);
			else
				sprintf(p->str, "%d", freq);

			font = gtk_style_get_font(GTK_WIDGET(wf)->style);

			p->strw = gdk_string_width(font, p->str);
			p->strh = gdk_string_height(font, p->str);
			p->strx = CLAMP(i - p->strw / 2, 0, width - p->strw);

			p->major = TRUE;
		}

		f += wf->resolution;

		p->next = list;
		list = p;
	}

	return list;
}

static void free_tics(struct tic *list)
{
	while (list) {
		struct tic *p = list;
		list = list->next;
		g_free(p);
	}
}
*/

static void draw_ruler(Waterfall *wf, HDC hDC)
{
	RECT	rect;
	int		iResolution, iMajorInc, iMinorInc, iFreq, iMin, iMax, iSel;
	int		iTickPosMajorTop, iTickPosMinorTop, iMarkerPosTop, iPosTickBottom;
	TCHAR	label[8];
	HGDIOBJ hOldPen, hOldFont, hOldBrush;
	POINT   aPoints[3];
	int     nWidth;

	switch (wf->fftlen) {
		case WATERFALL_FFTLEN_0:
			iMajorInc   = 500;
			iMinorInc   = 100;
			iResolution = 8389;
			break;
		case WATERFALL_FFTLEN_1:
			iMajorInc   = 200;
			iMinorInc   = 20;
			iResolution = 16777;
			break;
		case WATERFALL_FFTLEN_2:
			iMajorInc   = 100;
			iMinorInc   = 20;
			iResolution = 33554;
			break;
		case WATERFALL_FFTLEN_4:
			iMajorInc   = 50;
			iMinorInc   = 10;
			iResolution = 67109;
			break;
	}

	GetClientRect(wf->hWnd, &rect);
	iMin = ((wf->ruler_ref_x << 16) - (1 << 15)) / iResolution;
	iMax = (((wf->ruler_ref_x + rect.right - rect.left) << 16) - (1 << 15)) / iResolution;

	if (wf->config.direction) {
		int  iHeight     = rect.bottom - rect.top - RULER_HEIGHT;
		iPosTickBottom   = iHeight;
		iTickPosMinorTop = iHeight + 2;
		iTickPosMajorTop = iHeight + 5;
		iMarkerPosTop    = iHeight + 7;
		rect.top = rect.bottom - RULER_HEIGHT;
	} else {
		iPosTickBottom   = RULER_HEIGHT - 1;
		iTickPosMinorTop = RULER_HEIGHT - 3;
		iTickPosMajorTop = RULER_HEIGHT - 6;
		iMarkerPosTop    = RULER_HEIGHT - 8;
		rect.bottom      = rect.top + RULER_HEIGHT;
	}
	FillRect(hDC, &rect, GetStockObject(BLACK_BRUSH));

	hOldPen  = SelectObject(hDC, GetStockObject(WHITE_PEN));
	hOldFont = SelectObject(hDC, wf->hFont);
	SetTextColor(hDC, RGB(255, 255, 255));
	SetBkColor(hDC, RGB(0, 0, 0));
	iFreq = 0;
	aPoints[0].y = iPosTickBottom;
	aPoints[1].y = iTickPosMinorTop;
	while (iFreq < iMax) {
		if (iFreq >= iMin) {
			int iPos       = ((iFreq * iResolution + (1 << 15)) >> 16) - wf->ruler_ref_x;
			aPoints[0].x   = iPos;
			aPoints[1].x   = iPos;
			Polyline(hDC, aPoints, 2);
		}
		iFreq += iMinorInc;
	}

	iFreq = 0;
	aPoints[1].y = iTickPosMajorTop;
	while (iFreq < iMax) {
		if (iFreq >= iMin) {
			int iPos       = ((iFreq * iResolution + (1 << 15)) >> 16) - wf->ruler_ref_x;
			aPoints[0].x   = iPos;
			aPoints[1].x   = iPos;
			Polyline(hDC, aPoints, 2);
			_stprintf(label, _T("%d"), iFreq);
			ExtTextOut(hDC, iPos, iTickPosMajorTop, 0, NULL, label, _tcslen(label), NULL);
		}
		iFreq += iMajorInc;
	}

	iSel = ((s_aChannels[s_iChannelActive]->frequency * (iResolution >> FREQ_FRAC_BITS) + (1 << 15)) >> 16) - wf->ruler_ref_x;
	if (iSel >= 0 && iSel <= rect.right - rect.left) {
		aPoints[0].x = iSel;
		aPoints[1].x = iSel - 5;
		aPoints[2].x = iSel + 5;
		aPoints[0].y	 = iPosTickBottom;
		aPoints[1].y	 = iTickPosMajorTop;
		aPoints[2].y	 = iTickPosMajorTop;
		Polygon(hDC, aPoints, 3);
	}

	nWidth      = rect.right - rect.left;
	rect.left   = 0;
	rect.right  = RULER_HEIGHT;
	rect.top    = iPosTickBottom;
	rect.bottom = iPosTickBottom + RULER_HEIGHT;
	FillRect(hDC, &rect, GetStockObject(WHITE_BRUSH));
	rect.left   = nWidth - RULER_HEIGHT;
	rect.right  = nWidth;
	rect.top    = iPosTickBottom;
	rect.bottom = iPosTickBottom + RULER_HEIGHT;
	FillRect(hDC, &rect, GetStockObject(WHITE_BRUSH));
	
	SelectObject(hDC, GetStockObject(BLACK_PEN));
	hOldBrush = SelectObject(hDC, GetStockObject(BLACK_BRUSH));
	aPoints[0].x = 3;
	aPoints[1].x = RULER_HEIGHT - 5;
	aPoints[2].x = RULER_HEIGHT - 5;
	aPoints[0].y = iPosTickBottom + (RULER_HEIGHT >> 1);
	aPoints[1].y = iPosTickBottom + 1;
	aPoints[2].y = iPosTickBottom + RULER_HEIGHT - 1;
	Polygon(hDC, aPoints, 3);

	aPoints[0].x = nWidth - 4;
	aPoints[1].x = nWidth - RULER_HEIGHT + 4;
	aPoints[2].x = nWidth - RULER_HEIGHT + 4;
	aPoints[0].y = iPosTickBottom + (RULER_HEIGHT >> 1);
	aPoints[1].y = iPosTickBottom + 1;
	aPoints[2].y = iPosTickBottom + RULER_HEIGHT - 1;
	Polygon(hDC, aPoints, 3);
	SelectObject(hDC, hOldPen);
	SelectObject(hDC, hOldFont);
	SelectObject(hDC, hOldBrush);
}

static void draw_waterfall_line(Waterfall *wf, HDC hDC, int iRow, COLORREF *pData, int nLength)
{
	int    i;
	UCHAR *pPixels = (UCHAR*)wf->pBitmapBits;
	for (i = 0; i < nLength; ++ i) {
		COLORREF  clr  = pData[i];
		UCHAR    *pClr = (UCHAR*)(&clr) + 2;
		*pPixels++ = *pClr--;
		*pPixels++ = *pClr--;
		*pPixels++ = *pClr;
	}
	{
		HDC      hBmpDC  = CreateCompatibleDC(hDC);
		HGDIOBJ *pObjOld = SelectObject(hBmpDC, wf->hBitmap);
		BitBlt(hDC, 0, iRow, nLength, 1, hBmpDC, 0, 0, SRCCOPY);
		SelectObject(hBmpDC, pObjOld);
		DeleteDC(hBmpDC);
	}
}

static void draw_waterfall(Waterfall *wf, HDC hDC, BOOL bRefresh)
{
	RECT    rect;
	POINT   aPoints[2];
	int     nWidth, nHeight, nWidthBuf, nWidthFFT, iPos, i;
	HGDIOBJ hOldPen;

	GetClientRect(wf->hWnd, &rect);
	nWidth		= rect.right  - rect.left;
	nHeight		= rect.bottom - rect.top - RULER_HEIGHT;
	nWidthBuf	= wf->fftlen >> 1;
	nWidthFFT	= nWidth;
	iPos		= wf->pixbufptr - (bRefresh ? nHeight : 1) * nWidthBuf;

	if (iPos < 0)
		iPos += wf->pixbufsize;

	if (bRefresh) {
		draw_ruler(wf, hDC);
	} else {
		RECT rect;
		rect.left   = 0;
		rect.right  = nWidth;
		rect.top    = 0;
		rect.bottom = nHeight;
		if (wf->config.direction) {
			ScrollWindowEx(wf->hWnd, 0, -1, &rect, 0, 0, 0, 0);
		} else {
			rect.top    += RULER_HEIGHT;
			rect.bottom += RULER_HEIGHT;
			ScrollWindowEx(wf->hWnd, 0,  1, &rect, 0, 0, 0, 0);
		}
	}

	if (nWidthFFT > nWidthBuf)
		nWidthFFT = nWidthBuf;

	hOldPen = SelectObject(hDC, GetStockObject(BLACK_PEN));

	for (i = (bRefresh ? 0 : (nHeight - 1)); i < nHeight; ++ i) {
		int iRow = wf->config.direction ? i : (nHeight - i - RULER_HEIGHT);
		draw_waterfall_line(wf, hDC, iRow,
			wf->pixbuf + wf->ruler_ref_x + iPos, nWidthFFT);
		if (nWidthFFT < nWidth) {
			aPoints[0].x = nWidthFFT;
			aPoints[0].y = iRow;
			aPoints[1].x = nWidth;
			aPoints[1].y = iRow;
			Polyline(hDC, aPoints, 2);
		}
		iPos += nWidthBuf;
		if (iPos == wf->pixbufsize)
			iPos = 0;
	}

	SelectObject(hDC, hOldPen);

	if (! bRefresh) {
//		draw_markers(wf);
	}
}

static void draw_spectrum(Waterfall *wf, HDC hDC, BOOL bRefresh, BOOL peak)
{
/*
	GtkWidget *widget;
	GdkSegment seg[6];
	GdkPoint *pnt;
	gint i, h;
	struct tic *tics, *list;

	widget = GTK_WIDGET(wf);

	// clear window 
	gdk_draw_rectangle(wf->pixmap, widget->style->black_gc, TRUE,
			   0,
			   0,
			   widget->allocation.width,
			   widget->allocation.height);

	// draw grid
	h = widget->allocation.height - RULER_HEIGHT - 1;
	for (i = 0; i < 6; i++) {
		seg[i].x1 = 0;
		seg[i].x2 = widget->allocation.width - 1;
		seg[i].y1 = RULER_HEIGHT + i * h / 5;
		seg[i].y2 = RULER_HEIGHT + i * h / 5;
	}
	gdk_draw_segments(wf->pixmap, wf->grid_gc, seg, 6);

	// draw ruler
	list = build_tics(wf);
	tics = list;

	while (tics) {
		if (tics->major) {
			gdk_draw_line(wf->pixmap, wf->grid_gc,
				      tics->x, RULER_HEIGHT - 9,
				      tics->x, widget->allocation.height - 1);

			gdk_draw_string(wf->pixmap,
					gtk_style_get_font(widget->style),
					wf->grid_gc,
					tics->strx,
					tics->strh + 2,
					tics->str);
		} else {
			gdk_draw_line(wf->pixmap, wf->grid_gc,
				      tics->x, RULER_HEIGHT - 5,
				      tics->x, RULER_HEIGHT - 1);
		}
		tics = tics->next;
	}

	free_tics(list);

	// draw trace
	pnt = g_new(GdkPoint, widget->allocation.width);

	h = 1 + RULER_HEIGHT - widget->allocation.height;
	for (i = 0; i < widget->allocation.width; i++) {
		gint f = i + (gint) (wf->startfreq / wf->resolution);

		if (peak)
			wf->peakbuf[i] = MAX(wf->peakbuf[f], wf->specbuf[f]);
		else
			wf->peakbuf[i] = wf->specbuf[f];

		pnt[i].x = i;
		pnt[i].y = RULER_HEIGHT + h * wf->peakbuf[i];
	}
	gdk_draw_lines(wf->pixmap, wf->trace_gc, pnt, widget->allocation.width);

	// draw markers
	draw_markers(wf);

	// draw to screen
	gdk_draw_pixmap(widget->window,
			widget->style->base_gc[widget->state],
			wf->pixmap, 
			0, 0, 0, 0,
			widget->allocation.width,
			widget->allocation.height);

	g_free(pnt);
*/
}

static void draw_scope(Waterfall *wf, HDC hDC, BOOL bRefresh)
{
/*
	GtkWidget *widget;
	GdkPoint *pnt;
	GdkSegment *seg, *segp;
	gint w, h, i, nseg;

	widget = GTK_WIDGET(wf);

	w = widget->allocation.width;
	h = widget->allocation.height;

	pnt = g_new(GdkPoint, w);
	seg = g_new(GdkSegment, w / 8 + 2);

	// calculate grid
	for (segp = seg, nseg = 0, i = 0; i < w; i += w / 8, segp++, nseg++) {
		segp->x1 = i;
		segp->x2 = i;
		segp->y1 = h / 2 - 5;
		segp->y2 = h / 2 + 5;
	}
	segp->x1 = 0;
	segp->x2 = w - 1;
	segp->y1 = h / 2;
	segp->y2 = h / 2;
	segp++;
	nseg++;
	segp->x1 = w / 2;
	segp->x2 = w / 2;
	segp->y1 = 0;
	segp->y2 = h - 1;
	segp++;
	nseg++;

	// copy trace
	for (i = 0; i < w; i++) {
		pnt[i].x = i;
		pnt[i].y = wf->inbuf[i] * h / 2 + h / 2;
	}

	// clear window
	gdk_draw_rectangle(wf->pixmap,
			   widget->style->black_gc,
			   TRUE, 0, 0, 
			   widget->allocation.width, 
			   widget->allocation.height);

	// draw grid
	gdk_draw_segments(wf->pixmap, wf->grid_gc, seg, nseg);

	// draw trace
	gdk_draw_lines(wf->pixmap, wf->trace_gc, pnt, w);

	// draw to screen
	gdk_draw_pixmap(widget->window,
			widget->style->base_gc[widget->state],
			wf->pixmap, 
			0, 0, 0, 0,
			widget->allocation.width,
			widget->allocation.height);

	g_free(pnt);
	g_free(seg);
*/
}

static void draw(Waterfall *wf, HDC hDC, BOOL bRefresh)
{
	switch (wf->config.mode) {
	case WATERFALL_MODE_NORMAL:
		draw_waterfall(wf, hDC, bRefresh);
		break;
	case WATERFALL_MODE_SPECTRUM:
		draw_spectrum(wf, hDC, bRefresh, FALSE);
		break;
	case WATERFALL_MODE_SPECTRUM_PEAK:
		draw_spectrum(wf, hDC, bRefresh, TRUE);
		break;
	case WATERFALL_MODE_SCOPE:
		draw_scope(wf, hDC, bRefresh);
		break;
	default:
//		g_warning("Unknown waterfall mode: %d\n", wf->config.mode);
		break;
	}
}

static BOOL waterfall_set_data2(Waterfall *wf)
{
	int      nFFTLen  = wf->fftlen;
	int      nWidth   = nFFTLen >> 1;
	int      nFFTBits = 0;
	int      i;
	short    *aReal = (short*)malloc(2 * nFFTLen * sizeof(short) + nFFTLen * sizeof(COLORREF));
	short    *aImag = aReal + nFFTLen;
	COLORREF *aLoud = (COLORREF*)(aReal + (nFFTLen << 1));
//    kiss_fft_cpx cout[WATERFALL_FFTLEN_MAX];

	for (i = 0; i < nFFTLen; ++ i) {
		aReal[i] = wf->inbuf[i];
		aImag[i] = 0;
	}

//	fix_loud_kiss(aLoud, (short*)cout, nWidth, 0);

	window(aReal, nFFTLen);
	while (nFFTLen > 1) {
		++ nFFTBits;
		nFFTLen >>= 1;
	}
	fix_rfft(aReal, aImag, nFFTBits);
//    fix_fft(aReal, aImag, nFFTBits, 0);

//	fix_loud(aLoud, aReal, aImag, nWidth, 0);
	for (i = 0; i < nWidth; ++ i)
		wf->loudbuf[i] += aReal[i] * aReal[i] + aImag[i] * aImag[i];

	if (++ wf->iAverage < wf->nAverages) {
		free(aReal);
		return FALSE;
	}

	wf->iAverage = 0;
	for (i = 0; i < nWidth; ++ i) {
		int loud  = (db_from_ampl2_2(wf->loudbuf[i]) - (wf->config.iDecibelsThreshold << 16)) / wf->config.iDecibelsRange;
		int color = (loud * wf->nColorTableLen + 32768) >> 16;
		if (color >= wf->nColorTableLen)
			color = wf->nColorTableLen - 1;
		else if (color < 0)
			color = 0;
		aLoud[i] = wf->aColorTable[color];
		wf->loudbuf[i] = 0;
	}

	if (wf->pixbufptr + nWidth <= wf->pixbufsize) {
		COLORREF *ptr = wf->pixbuf + wf->pixbufptr;
		for (i = 0; i < nWidth; ++ i)
			*ptr++ = aLoud[i];
		wf->pixbufptr += nWidth;
		if (wf->pixbufptr == wf->pixbufsize)
			wf->pixbufptr = 0;
	} else {
		COLORREF *ptr  = wf->pixbuf + wf->pixbufptr;
		int	      nLen = wf->pixbufsize - wf->pixbufptr;
		for (i = 0; i < nLen; ++ i)
			*ptr++ = aLoud[i];
		ptr = 0;
		for (; i < nWidth; ++ i)
			*ptr++ = aLoud[i];
		wf->pixbufptr = nWidth - nLen;
	}

	free(aReal);
	return TRUE;
}

void waterfall_set_data(Waterfall *wf, short *data, int len)
{
	int overlap = 0;
	switch (wf->config.speed) {
	case WATERFALL_SPEED_HALF:
		overlap = 2048;
		break;
	case WATERFALL_SPEED_NORMAL:
		overlap = 1024;
		break;
	case WATERFALL_SPEED_DOUBLE:
		overlap = 512;
		break;
	case WATERFALL_SPEED_TRIPLE:
		overlap = 341;
		break;
	}

	while (len-- > 0) {
		wf->inbuf[wf->inptr ++] = *data++;
		if (wf->inptr >= wf->fftlen) {
			int i;
			if (wf->pixbuf != NULL && waterfall_set_data2(wf)) {
				HDC hDC = GetDC(wf->hWnd);
				if (++ wf->ruler_update_cntr > 10) {
					draw(wf, hDC, TRUE);
					wf->ruler_update_cntr = 0;
				} else 
					draw(wf, hDC, FALSE);
				ReleaseDC(wf->hWnd, hDC);
			}
			for (i = 0; i < wf->inptr - overlap; i++)
				wf->inbuf[i] = wf->inbuf[i +overlap];
			wf->inptr -= overlap;
		}
	}
}

LRESULT CALLBACK waterfall_wndproc(HWND hWnd, UINT message, WPARAM wParam, LPARAM lParam)
{
	Waterfall *wf = (Waterfall*)GetWindowLong(hWnd, GWL_USERDATA);
	if (message == WM_PAINT) {
		PAINTSTRUCT paint;
		draw(wf, BeginPaint(hWnd, &paint), TRUE);
		EndPaint(hWnd, &paint);
	} else if (message == WM_ERASEBKGND) {
		// ignore background erasing
		return TRUE;
	} else if (message == WM_LBUTTONDOWN) {
		RECT rect;
		GetClientRect(wf->hWnd, &rect);
		wf->bMouseButtonDown = TRUE;
		wf->MousePrevPos.x = LOWORD(lParam);
		wf->MousePrevPos.y = HIWORD(lParam);
		wf->nMouseMode     = 0;
		SetCapture(hWnd);
		if (wf->config.direction) {
			if (wf->MousePrevPos.y >= rect.bottom - rect.top - RULER_HEIGHT)
				wf->nMouseMode = 1;
		} else if (wf->MousePrevPos.y <= RULER_HEIGHT)
			wf->nMouseMode = 1;
		if (wf->nMouseMode == 0) {
			int iResolution, iFreq;
			switch (wf->fftlen) {
			case WATERFALL_FFTLEN_0:	iResolution = 512000;	break;
			case WATERFALL_FFTLEN_1:	iResolution = 256000;	break;
			case WATERFALL_FFTLEN_2:	iResolution = 128000;	break;
			case WATERFALL_FFTLEN_4:	iResolution =  64000;	break;
			}
			iFreq = ((wf->ruler_ref_x + wf->MousePrevPos.x) * iResolution +
				(1 << (15 - FREQ_FRAC_BITS))) >> (16 - FREQ_FRAC_BITS);
			s_aChannels[s_iChannelActive]->frequency = iFreq;
			waterfall_draw_ruler(wf, 0);
		} else if (wf->nMouseMode == 1 &&
			(wf->MousePrevPos.x <= RULER_HEIGHT || 
			 wf->MousePrevPos.x >= rect.right - rect.left - RULER_HEIGHT)) {
			int nWidth    = rect.right - rect.left;
			int nRulerMax = (wf->fftlen >> 1) - nWidth;
			wf->nMouseMode = 0;
			if (wf->MousePrevPos.x <= RULER_HEIGHT)
				wf->ruler_ref_x -= nWidth / 3;
			else
				wf->ruler_ref_x += nWidth / 3;
			if (wf->ruler_ref_x > nRulerMax)
				wf->ruler_ref_x = nRulerMax;
			if (wf->ruler_ref_x < 0)
				wf->ruler_ref_x = 0;
			{
				HDC hDC = GetDC(wf->hWnd);
				draw(wf, hDC, TRUE);
				ReleaseDC(wf->hWnd, hDC);
			}
		}
	} else if (message == WM_SIZE) {
		if (wf != 0) {
			alloc_pixbuf(wf, FALSE);
			InvalidateRect(wf->hWnd, 0, FALSE);
		}
	} else if (message == WM_MOUSEMOVE) {
        POINT pt;
		pt.x = LOWORD(lParam);
		pt.y = HIWORD(lParam);
		if (wf->bMouseButtonDown) {
			if (wf->nMouseMode == 1) {
				RECT  rect;
				int   nRulerMax;
				wf->ruler_ref_x += wf->MousePrevPos.x - pt.x;
				GetClientRect(hWnd, &rect);
				nRulerMax = (wf->fftlen >> 1) - rect.right + rect.left;
				if (wf->ruler_ref_x > nRulerMax)
					wf->ruler_ref_x = nRulerMax;
				if (wf->ruler_ref_x < 0)
					wf->ruler_ref_x = 0;
				{
					HDC hDC = GetDC(wf->hWnd);
					draw(wf, hDC, TRUE);
					ReleaseDC(wf->hWnd, hDC);
				}
			}
		}
		wf->MousePrevPos.x = pt.x;
		wf->MousePrevPos.y = pt.y;
	} else if (message == WM_LBUTTONUP) {
		wf->bMouseButtonDown = FALSE;
		wf->nMouseMode       = 0;
		ReleaseCapture();
	}
	return DefWindowProc(hWnd, message, wParam, lParam);
}

void waterfall_draw_ruler(Waterfall *wf, HDC hDC)
{
	if (hDC == 0) {
		HDC hDC = GetDC(wf->hWnd);
		draw_ruler(wf, hDC);
		ReleaseDC(wf->hWnd, hDC);
	} else {
		draw_ruler(wf, hDC);
	}
}
