/*
 *    trx.c  --  Modem engine
 *
 *    Copyright (C) 2001, 2002, 2003
 *      Tomi Manninen (oh2bns@sral.fi)
 *
 *    This file is part of gMFSK.
 *
 *    gMFSK is free software; you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation; either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    gMFSK is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with gMFSK; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

#include <windows.h>
#include <tchar.h>

#include "trx.h"
#include "sndmodem.h"
#include "qsodata.h"
#include "ascii.h"
#include "../misc/misc.h"
#include "../misc/mixer.h"
#include "../resource.h"

#ifdef _WIN32_WCE
#include <Aygshell.h>
#else
#include <stdio.h>
#endif /* _WIN32_WCE */

extern void mfsk_init(Trx *trx);
extern void rtty_init(Trx *trx);
extern void throb_init(Trx *trx);
extern void psk31_init(Trx *trx);
extern void cw_init(Trx *trx);
extern void mt63_init(Trx *trx);
extern void feld_init(Trx *trx);

/* ---------------------------------------------------------------------- */

TCHAR *trx_mode_names[] = {
	_T("BPSK31"),
	_T("QPSK31"),
	_T("PSK63"),
	_T("RTTY"),
	_T("CW"),
	_T("MFSK16"),
	_T("MFSK8"),
	_T("THROB1"),
	_T("THROB2"),
	_T("THROB4"),
	_T("MT63"),
	_T("FELDHELL"),
	_T("FMHELL")
};

TCHAR *trx_state_names[] = {
	_T("PAUSED"),
	_T("RECEIVE"),
	_T("TRANSMIT"),
	_T("TUNING"),
	_T("ABORTED"),
	_T("FLUSHING")
};

/// Forward reference to the trx window procedure
LRESULT CALLBACK trx_wndproc(HWND, UINT, WPARAM, LPARAM);
LRESULT CALLBACK rx_wndproc(HWND, UINT, WPARAM, LPARAM);
LRESULT CALLBACK tx_wndproc(HWND, UINT, WPARAM, LPARAM);

static const TBYTE	WIN32CLASS_TRX[]	= _T("USR_TRX");
static const TBYTE	WIN32CLASS_RX[]		= _T("USR_RX");
static const TBYTE	WIN32CLASS_TX[]		= _T("USR_TX");
static WNDPROC		edit_wndproc		= 0;

BOOL trx_register_classes(HINSTANCE hInstance)
{
  WNDCLASS	wc, wcEdit;

  wc.style			= CS_HREDRAW | CS_VREDRAW;
  wc.lpfnWndProc	= (WNDPROC) trx_wndproc;
  wc.cbClsExtra		= 0;
  wc.cbWndExtra		= 0;
  wc.hInstance		= hInstance;
  wc.hIcon			= 0;
  wc.hCursor		= 0;
  wc.hbrBackground	= (HBRUSH) GetStockObject(WHITE_BRUSH);
  wc.lpszMenuName	= 0;
  wc.lpszClassName	= WIN32CLASS_TRX;

  if (RegisterClass(&wc) == 0)
	  return FALSE;

  if (! GetClassInfo(hInstance, _T("EDIT"), &wcEdit)) 
	  return FALSE;

  memcpy(&wc, &wcEdit, sizeof(wc));
  wc.lpszClassName  = WIN32CLASS_RX;
  edit_wndproc		= wc.lpfnWndProc;
  wc.lpfnWndProc	= (WNDPROC)rx_wndproc;
  if (RegisterClass(&wc) == 0)
	  return FALSE;

  wc.lpszClassName  = WIN32CLASS_TX;
  wc.lpfnWndProc	= (WNDPROC)tx_wndproc;
  if (RegisterClass(&wc) == 0)
	  return FALSE;

  return TRUE;
}

Trx* trx_new()
{
	Trx *trx = (Trx*)malloc(sizeof(Trx));
	if (trx == 0)
		return 0;
	if (! trx_init(trx)) {
		trx_destroy(trx);
		return 0;
	}
	trx_initmodem(trx);
	return trx;
}

void trx_free(Trx *trx)
{
	if (trx == 0)
		return;
	trx_destroy(trx);
	free(trx);
}

BOOL trx_init(Trx *trx)
{
	memset(trx, 0, sizeof(Trx));
	trx->mode		= MODE_BPSK31;
//	trx->mode		= MODE_QPSK31;
//	trx->mode		= MODE_MFSK16;
//	trx->mode		= MODE_RTTY;
	trx->state		= TRX_STATE_RX;
//	trx.samplerate	= 8000;
//	trx->afcon		= TRUE;
	trx->squelchon	= FALSE;
	trx->iSquelchThreshold = 50;
	trx->reverse	= FALSE;
	trx->frequency	= 2000 << FREQ_FRAC_BITS;
	trx->cw_rxspeed   = 4; // 10 WPM
	trx->cw_txspeed   = 18;
	trx->cw_bandwidth = 3; // 70 Hz
	trx->cw_firtaps   = 0; // 64 taps
	//FIXME
	trx_set_rtty_parms(trx, /* squelch */0,  170 << FREQ_FRAC_BITS, 45.45, 0 /* 5 bits */, FALSE, 1 /* 1.5 */,
			trx->reverse, FALSE /* msbfirst */);

	trx->pQso = qsodata_new();
	mixer_initialize();
	return TRUE;
}

BOOL trx_initmodem(Trx *trx)
{
	trx->state = TRX_STATE_RX;

	if (trx->destructor != 0) {
		trx->destructor(trx);
		trx->txinit		= 0;
		trx->rxinit		= 0;
		trx->txprocess	= 0;
		trx->rxprocess	= 0;
		trx->destructor = 0;
	}

	switch (trx->mode) {
	case MODE_RTTY:
		rtty_init(trx);
		break;

	case MODE_BPSK31:
	case MODE_QPSK31:
	case MODE_PSK63:
		psk31_init(trx);
		break;

	case MODE_CW:
		cw_init(trx);
		break;

	case MODE_MFSK16:
	case MODE_MFSK8:
		mfsk_init(trx);
		break;

/*
	case MODE_THROB1:
	case MODE_THROB2:
	case MODE_THROB4:
		throb_init(trx);
		break;

	case MODE_MT63:
		mt63_init(trx);
		break;

	case MODE_FELDHELL:
	case MODE_FMHELL:
		feld_init(trx);
		break;
*/
	}

	trx->rxinit(trx);

	return TRUE;
}

BOOL trx_create(Trx *trx, HWND hWndParent, HANDLE hInstance, SndModem *pModem)
{
	trx->hWnd = CreateWindow(WIN32CLASS_TRX, NULL, WS_CHILD | WS_CLIPCHILDREN | WS_CLIPSIBLINGS | WS_VISIBLE,
		0, 0, 10, 10, hWndParent, NULL, hInstance, (void*)trx);
	if (! trx->hWnd)
		return FALSE;
	trx->hWndReceive = CreateWindow(WIN32CLASS_RX, NULL, 
		WS_BORDER | WS_CHILD | WS_CLIPSIBLINGS | WS_VISIBLE | WS_VSCROLL | 
		ES_AUTOVSCROLL | ES_MULTILINE,
		0, 0, 10, 10, trx->hWnd, NULL, hInstance, (void*)trx);
	trx->hWndSend    = CreateWindow(WIN32CLASS_TX, NULL, 
		WS_BORDER | WS_CHILD | WS_CLIPSIBLINGS | WS_VISIBLE | WS_VSCROLL |
		ES_AUTOVSCROLL | ES_MULTILINE | ES_WANTRETURN,
		0, 0, 10, 10, trx->hWnd, NULL, hInstance, (void*)trx);

	trx->pModem = pModem;
	
	{
		LOGFONT logfont;
		memset(&logfont, 0, sizeof(LOGFONT));
		logfont.lfHeight  = -11;
		logfont.lfWeight  = FW_NORMAL;
		logfont.lfCharSet = ANSI_CHARSET;
		_tcscpy(logfont.lfFaceName, _T("Tahoma"));
		trx->hFont = CreateFontIndirect(&logfont);
	}

	SendMessage(trx->hWnd,			WM_SETFONT, (WPARAM)trx->hFont, FALSE);
	SendMessage(trx->hWndReceive,	WM_SETFONT, (WPARAM)trx->hFont, FALSE);
	SendMessage(trx->hWndSend,		WM_SETFONT, (WPARAM)trx->hFont, FALSE);
	SendMessage(trx->hWndReceive,	EM_SETLIMITTEXT, 64000, FALSE);
	SendMessage(trx->hWndSend,		EM_SETLIMITTEXT, 64000, FALSE);
	SendMessage(trx->hWndReceive,	EM_SETREADONLY, (WPARAM)FALSE, 0);
	
	SetWindowLong(trx->hWnd, GWL_USERDATA, (long)trx);

	return TRUE;
}

void trx_destroy(Trx *trx)
{
	SetWindowLong(trx->hWnd, GWL_USERDATA, 0l);
	DeleteObject(trx->hFont);
	DestroyWindow(trx->hWndReceive);
	DestroyWindow(trx->hWndSend);
	DestroyWindow(trx->hWnd);
	trx->destructor(trx);
	qsodata_destroy(trx->pQso);
	log_to_file_activate(&trx->log, FALSE, 0);
	memset(trx, 0, sizeof(Trx));
}

BOOL trx_layout(Trx *trx, layout_t nLayout, RECT *rect)
{
	HDC			hDC		 = GetWindowDC(trx->hWnd);
	HGDIOBJ		hFontOld = SelectObject(hDC, trx->hFont);
	int		    nWidth   = rect->right - rect->left;
	int		    nHeight  = rect->bottom - rect->top;
	TEXTMETRIC  tm;

	trx->nLayout = nLayout;

	GetTextMetrics(hDC, &tm);

	if (nLayout == LAYOUT_SINGLELINE) {
		rect->bottom = rect->top + tm.tmHeight;
		ShowWindow(trx->hWndReceive, SW_HIDE);
		ShowWindow(trx->hWndSend, SW_HIDE);
	} else if (nLayout == LAYOUT_RCV) {
		ShowWindow(trx->hWndSend, SW_HIDE);
		MoveWindow(trx->hWndReceive, -1, 0, nWidth + 2, nHeight + 1, TRUE);
		ShowWindow(trx->hWndReceive, SW_SHOW);
	} else if (nLayout == LAYOUT_TRX) {
		int nHalf = nHeight / 2;
		MoveWindow(trx->hWndReceive, -1, 0, nWidth + 2, nHalf + 1, TRUE);
		ShowWindow(trx->hWndReceive, SW_SHOW);
//		UpdateWindow(trx->hWndReceive);
		MoveWindow(trx->hWndSend, -1, nHalf, nWidth + 2, nHeight - nHalf + 1, TRUE);
		ShowWindow(trx->hWndSend, SW_SHOW);
//		SetWindowPos(trx->hWndSend, HWND_TOP, 0, nHalf, nWidth, nHeight - nHalf, SWP_SHOWWINDOW);
//		UpdateWindow(trx->hWndSend);
//		SetFocus(trx->hWndSend);
	}

	SelectObject(hDC, hFontOld);
	ReleaseDC(trx->hWnd, hDC);
	return TRUE;
}

static void trx_display_rxchar(Trx *trx, UINT data, BOOL bIsEcho)
{
	int nLength = SendMessage(trx->hWndReceive, WM_GETTEXTLENGTH, 0, 0);

/*
#ifndef _WIN32_WCE
	{
		char bufout[256];
		sprintf(bufout, "Char received: %d: '%c'\n", data, data);
		OutputDebugString(bufout);
	}
#endif /* _WIN32_WCE */

/*
	if (trx->bPrevCharEcho != bIsEcho &&
		trx->cPrevChar != '\r' && trx->cPrevChar != '\n' && data != '\r' && data != '\n') {
		// Switching between RX/TX. Add crlf if it has not been 
		TCHAR buf[2] = { '\r', 0 };
		SendMessage(trx->hWndReceive, EM_REPLACESEL, (WPARAM)FALSE, (LPARAM)&buf);
	}
	
	trx->bPrevCharEcho = bIsEcho;
*/

	if (data == '\r' || (trx->cPrevChar != '\r' && data == '\n')) {
		// send crlf
		TCHAR buf[3] = { '\r', '\n', 0 };
		SendMessage(trx->hWndReceive, EM_REPLACESEL, (WPARAM)FALSE, (LPARAM)&buf);
		trx->cPrevChar = data;
		trx->bPrevCharEcho = bIsEcho;
		return;
	}

	if (trx->cPrevChar == '\r' && data == '\n') {
		// crlf was already entered into the receive window
		trx->cPrevChar     = '\n';
		trx->bPrevCharEcho = bIsEcho;
		return;
	}

	if (trx->bPrevCharEcho != bIsEcho && trx->cPrevChar != '\r' && trx->cPrevChar != '\n') {
		// Switching between RX/TX. Add crlf if it has not been added yet.
		TCHAR buf[3] = { '\r', '\n', 0 };
		SendMessage(trx->hWndReceive, EM_REPLACESEL, (WPARAM)FALSE, (LPARAM)&buf);
	}

	trx->bPrevCharEcho = bIsEcho;
	trx->cPrevChar = data;

	if (data == '\b') {
		if (nLength > 0) {
			UINT empty = 0;
			SendMessage(trx->hWndReceive, EM_SETSEL, nLength - 1, nLength);
			SendMessage(trx->hWndReceive, EM_REPLACESEL, (WPARAM)FALSE /* can undo */, (LPARAM)&empty);
		}
	} else {
		SendMessage(trx->hWndReceive, EM_SETSEL, SendMessage(trx->hWndReceive, WM_GETTEXTLENGTH, 0, 0), -1);
		if (data < 32) {
			SendMessage(trx->hWndReceive, EM_REPLACESEL, (WPARAM)FALSE /* can undo */, (LPARAM)ascii[data]);
		} else {
			TCHAR buf[2] = { data, 0 };
			SendMessage(trx->hWndReceive, EM_REPLACESEL, (WPARAM)FALSE /* can undo */, (LPARAM)&buf);
		}
	}

	if (trx->nLayout == LAYOUT_SINGLELINE) {
		HDC			hDC      = GetDC(trx->hWnd);
		HGDIOBJ		hFontOld = SelectObject(hDC, trx->hFont);
		RECT		rect, rectWnd, rectClip;
		TCHAR       label[3] = _T("0:");
		TCHAR		newchar[2] = _T("0");
		int         nChannelLabelWidth, nNewCharWidth, nShift;

		// Get window rectangle
		GetClientRect(trx->hWnd, &rectWnd);
		memcpy(&rect, &rectWnd, sizeof(rect));

		// Compute width of the channel label
		label[0] += trx->nChannelID;
		DrawText(hDC, label, 2, &rect, DT_CALCRECT | DT_NOCLIP | DT_NOPREFIX | DT_LEFT | DT_SINGLELINE);
		nChannelLabelWidth = rect.right - rect.left;

		// Compute width of the new character
		memcpy(&rect, &rectWnd, sizeof(rect));
		newchar[0] = data;
		DrawText(hDC, newchar, 1, &rect, DT_CALCRECT | DT_NOCLIP | DT_NOPREFIX | DT_LEFT | DT_SINGLELINE);
		nNewCharWidth = rect.right - rect.left;

		nShift = rectWnd.right - rectWnd.left - (trx->nLineOutPos + nNewCharWidth);
		if (nShift < 0) {
			// Scroll the text (without the channel label)
			memcpy(&rect, &rectWnd, sizeof(rect));
			memcpy(&rectClip, &rectWnd, sizeof(rect));
			rect.left     = nChannelLabelWidth;
			rectClip.left = nChannelLabelWidth;
			ScrollWindowEx(trx->hWnd, nShift, 0, 0, &rectClip, 0, 0, 0);
			trx->nLineOutPos = rectWnd.right - rectWnd.left - nNewCharWidth;
		}
		// Draw the new character
		memcpy(&rect, &rectWnd, sizeof(rect));
		rect.left = trx->nLineOutPos;
		DrawText(hDC, newchar, 1, &rect, DT_NOCLIP | DT_NOPREFIX | DT_LEFT | DT_SINGLELINE);
		trx->nLineOutPos += nNewCharWidth;

		SelectObject(hDC, hFontOld);
		ReleaseDC(trx->hWnd, hDC);
	}

/*
	static int iOldPosRight = 0;
	HDC hDC = ::GetDC(s_hWndMain);
	bool bScroll = false;
	RECT rectLine;
	rectLine.left   = iOldPosRight;
	rectLine.right  = 300;
	rectLine.bottom = 320 - 2 * MENU_HEIGHT;
	rectLine.top    = rectLine.bottom - 20;
	if (data == 13 || data == 10)
		bScroll = true;
	else {
		::DrawText(hDC, (TCHAR*)&data, 1, &rectLine, DT_BOTTOM | DT_LEFT | DT_CALCRECT | DT_SINGLELINE);
		if (rectLine.right > 240) {
			rectLine.right	= rectLine.right - rectLine.left;
			rectLine.left	= 0;
			bScroll			= true;
		}
	}
	if (bScroll) {
		RECT rect;
		rect.left   = 0;
		rect.top    = 180;
		rect.right  = 240;
		rect.bottom = 320 - 2 * MENU_HEIGHT;
		::ScrollWindowEx(s_hWndMain, 0, -20, &rect, 0, 0, 0, SW_ERASE | SW_INVALIDATE);
		::UpdateWindow(s_hWndMain);
	}
	if (data != 13 && data != 10)
		::DrawText(hDC, (TCHAR*)&data, 1, &rectLine, DT_BOTTOM | DT_LEFT | DT_SINGLELINE);
	::ReleaseDC(s_hWndMain, hDC);

	iOldPosRight = rectLine.right;
*/
}

void trx_put_rx_char(Trx *trx, UINT c)
{
	trx_display_rxchar(trx, c, FALSE);
	if (c < 32) {
		log_to_file(&trx->log, LOG_RX, ascii[c]);
	} else {
		TCHAR buf[2] = { c, 0 };
		log_to_file(&trx->log, LOG_RX, buf);
	}
}

void trx_put_echo_char(Trx *trx, UINT c)
{
	trx_display_rxchar(trx, c, TRUE);
	if (c < 32) {
		log_to_file(&trx->log, LOG_TX, ascii[c]);
	} else {
		TCHAR buf[2] = { c, 0 };
		log_to_file(&trx->log, LOG_TX, buf);
	}
}

void trx_send_char(Trx *trx, TCHAR c)
{
	int   nLength = SendMessage(trx->hWndSend, WM_GETTEXTLENGTH, 0, 0);
	TCHAR data[2] = { c, 0 };
	SendMessage(trx->hWndSend, EM_SETSEL, SendMessage(trx->hWndSend, WM_GETTEXTLENGTH, 0, 0), -1);
	SendMessage(trx->hWndSend, EM_REPLACESEL, (WPARAM)FALSE /* can undo */, (LPARAM)data);
}

void trx_send_string(Trx *trx, TCHAR *str)
{
	int nLength = SendMessage(trx->hWndSend, WM_GETTEXTLENGTH, 0, 0);
	SendMessage(trx->hWndSend, EM_SETSEL, SendMessage(trx->hWndSend, WM_GETTEXTLENGTH, 0, 0), -1);
	SendMessage(trx->hWndSend, EM_REPLACESEL, (WPARAM)FALSE /* can undo */, (LPARAM)str);
}

void trx_clearrx(Trx *trx)
{
	SetWindowText(trx->hWndReceive, _T(""));
	trx->cPrevChar		= 0;
	trx->bPrevCharEcho	= FALSE;
}

void trx_cleartx(Trx *trx)
{
	SetWindowText(trx->hWndSend, _T(""));
	trx->nCharsSent = 0;
}

BOOL trx_is_sending(Trx *trx)
{
	return trx->state != TRX_STATE_PAUSE && trx->state != TRX_STATE_RX;
}

BOOL trx_txloop(Trx *trx)
{
	while (sndmodem_tx_buffer_underflowed(trx->pModem)) {
		if (trx->txprocess(trx) < 0) {
			trx->state = TRX_STATE_FLUSH;
			break;
		}
	}
	return TRUE;
//	clear_tx_text();
}

void trx_txend(Trx *trx)
{
	trx->backspaces = 0;
	trx->stopflag = 0;
	trx->tune = 0;
	trx->state = TRX_STATE_RX;
}

int trx_get_tx_char(Trx *trx)
{
	//FIXME
/*
	static int iChar = 'a';
	int iRes = iChar;
	if (iChar == 'z' || iChar == -1)
		iChar = -1;
	else
		++ iChar;
	return iChar;
*/
	int     nLength = SendMessage(trx->hWndSend, WM_GETTEXTLENGTH, 0, 0);
	LPCTSTR lpszText;
	HANDLE  hData;
	TCHAR   iChar   = -1;
	if (nLength <= trx->nCharsSent) {
		trx->nCharsSent = nLength;
		return -1;
	}
	SendMessage(trx->hWndSend, EM_SETSEL, trx->nCharsSent, trx->nCharsSent + 1);
	SendMessage(trx->hWndSend, WM_COPY, 0, 0);
	SendMessage(trx->hWndSend, EM_SETSEL, nLength, -1);
	OpenClipboard(NULL);
	hData = GetClipboardData(
#ifdef _WIN32_WCE
		CF_UNICODETEXT
#else /* _WIN32_WCE */
		CF_TEXT
#endif /* _WIN32_WCE */
		);
	if (hData != 0) {
		lpszText = (LPTSTR)GlobalLock(hData);
		iChar = lpszText[0];
		GlobalUnlock(hData);
    }
	EmptyClipboard();
	CloseClipboard();
	++ trx->nCharsSent;
	if (iChar == TRX_RX_CMD || iChar == TRX_CLEAR_CMD) {
		s_aChannels[s_iChannelActive]->stopflag = TRUE;
		if (iChar == TRX_CLEAR_CMD)
			trx_cleartx(trx);
		return -1;
	}
	return iChar;
}

static void draw(Trx *trx, HDC hDC)
{
	if (trx->nLayout == LAYOUT_SINGLELINE) {
		HGDIOBJ		hFontOld = SelectObject(hDC, trx->hFont);
		RECT		rectWnd, rect;
		TCHAR       label[3] = _T("0:");
		int         nTextLength = SendMessage(trx->hWndReceive, WM_GETTEXTLENGTH, 0, 0);
		TCHAR      *pText = malloc(sizeof(TCHAR) * nTextLength + 1);
		int         nChannelLabelWidth;

		// Get window rectangle
		GetClientRect(trx->hWnd, &rectWnd);
		memcpy(&rect, &rectWnd, sizeof(rect));

		// Compute width of the channel label
		label[0] += trx->nChannelID;
		DrawText(hDC, label, 2, &rect, DT_CALCRECT | DT_NOCLIP | DT_NOPREFIX | DT_LEFT | DT_SINGLELINE);
		nChannelLabelWidth = rect.right - rect.left;

		SendMessage(trx->hWndReceive, WM_GETTEXT, (WPARAM)(nTextLength + 1), (LPARAM)pText); 
		memcpy(&rect, &rectWnd, sizeof(rect));
		DrawText(hDC, pText, nTextLength, &rect, DT_CALCRECT | DT_NOCLIP | DT_NOPREFIX | DT_RIGHT | DT_SINGLELINE);
		if (rect.right - rect.left < rectWnd.right - rectWnd.left - nChannelLabelWidth) {
			trx->nLineOutPos = rect.right - rect.left;
			memcpy(&rect, &rectWnd, sizeof(rect));
			rect.left = nChannelLabelWidth;
			DrawText(hDC, pText, nTextLength, &rect, DT_NOCLIP | DT_NOPREFIX | DT_LEFT | DT_SINGLELINE);
		} else {
			trx->nLineOutPos = rectWnd.right;
			memcpy(&rect, &rectWnd, sizeof(rect));
			rect.left = nChannelLabelWidth;
			DrawText(hDC, pText, nTextLength, &rect, DT_NOCLIP | DT_NOPREFIX | DT_RIGHT | DT_SINGLELINE);
		}
		free(pText);

		memcpy(&rect, &rectWnd, sizeof(rect));
		SetTextColor(hDC, RGB(255, 255, 255));
		SetBkColor(hDC, RGB(0, 0, 0));
		DrawText(hDC, label, 2, &rect, DT_NOCLIP | DT_NOPREFIX | DT_LEFT | DT_SINGLELINE);

		SelectObject(hDC, hFontOld);
	}
}

LRESULT CALLBACK trx_wndproc(HWND hWnd, UINT message, WPARAM wParam, LPARAM lParam)
{
	Trx *trx = (Trx*)GetWindowLong(hWnd, GWL_USERDATA);
	if (trx == 0)
		return DefWindowProc(hWnd, message, wParam, lParam);

	if (message == WM_PAINT) {
		PAINTSTRUCT paint;
		draw(trx, BeginPaint(hWnd, &paint));
		EndPaint(hWnd, &paint);
		return 0;
	}
	if (message == WM_LBUTTONDOWN) {
		SendMessage(GetParent(hWnd), WM_COMMAND, (WPARAM)(ID_CHANNEL0 + trx->nChannelID), 0);
		return 0;
	}
	return DefWindowProc(hWnd, message, wParam, lParam);
}

void ShowContextMenu(HWND hWnd, int x, int y, BOOL bIsTX)
{
	HMENU hMenu = CreatePopupMenu();
	int   iCmd;
	POINT pt;
	int   nStartChar, nEndChar;
	int	  nLength		  = GetWindowTextLength(hWnd);
	UINT  uiFlagsEmpty    = MF_ENABLED | MF_STRING;
	UINT  uiFlagsEmptySel = MF_ENABLED | MF_STRING;
	Trx  *trx			  = (Trx*)GetWindowLong(GetParent(hWnd), GWL_USERDATA);
	HANDLE hClipboardData = 0;

	pt.x = x;
	pt.y = y;
	ClientToScreen(hWnd, &pt);

	SendMessage(hWnd, EM_GETSEL, (WPARAM)&nStartChar, (LPARAM)&nEndChar);

	OpenClipboard(NULL);
	hClipboardData = GetClipboardData(
#ifdef _WIN32_WCE
		CF_UNICODETEXT
#else /* _WIN32_WCE */
		CF_TEXT
#endif /* _WIN32_WCE */
		);
	CloseClipboard();

	if (nLength == 0)
		uiFlagsEmpty    |= MF_GRAYED;
	if (nStartChar >= nEndChar)
		uiFlagsEmptySel |= MF_GRAYED;

	AppendMenu(hMenu, uiFlagsEmpty,				ID_CLEAR,		_T("Clear Window"));
	AppendMenu(hMenu, uiFlagsEmptySel,			ID_COPY,		_T("Copy"));
	AppendMenu(hMenu, uiFlagsEmptySel,			ID_CUT,			_T("Cut"));
	AppendMenu(hMenu, MF_ENABLED | MF_STRING | (hClipboardData ? 0 : MF_GRAYED),	
												ID_PASTE,		_T("Paste"));
	AppendMenu(hMenu, uiFlagsEmpty,				ID_SELECT_ALL,	_T("Select All"));

	if (! bIsTX && nStartChar < nEndChar) {
		AppendMenu(hMenu, MF_ENABLED | MF_SEPARATOR, 0, 0);
		AppendMenu(hMenu, uiFlagsEmptySel, ID_COPY_CALL,		_T("Call"));
		AppendMenu(hMenu, uiFlagsEmptySel, ID_COPY_NAME,		_T("Name"));
		AppendMenu(hMenu, uiFlagsEmptySel, ID_COPY_QTH,			_T("QTH"));
		AppendMenu(hMenu, uiFlagsEmptySel, ID_COPY_LOC,			_T("Loc"));
		AppendMenu(hMenu, uiFlagsEmptySel, ID_COPY_RST_SENT,	_T("TX RST"));
		AppendMenu(hMenu, uiFlagsEmptySel, ID_COPY_RST_RCVD,	_T("RX RST"));
		AppendMenu(hMenu, uiFlagsEmptySel, ID_COPY_NOTES,		_T("Notes"));
	}

	iCmd = TrackPopupMenuEx(hMenu, TPM_LEFTALIGN | TPM_RETURNCMD, pt.x, pt.y, hWnd, 0);
	switch (iCmd) {
	case ID_CLEAR:
		if (bIsTX)
			trx_cleartx(trx);
		else
			trx_clearrx(trx);
		break;
	case ID_COPY:
		SendMessage(hWnd, WM_COPY, 0, 0);
		break;
	case ID_CUT:
		SendMessage(hWnd, WM_CUT, 0, 0);
		break;
	case ID_PASTE:
		SendMessage(hWnd, WM_PASTE, 0, 0);
		break;
	case ID_SELECT_ALL:
		SendMessage(hWnd, EM_SETSEL, (WPARAM)(INT)0, (LPARAM)(INT)-1);
		break;
	case ID_COPY_CALL:
	case ID_COPY_NAME:
	case ID_COPY_QTH:
	case ID_COPY_LOC:
	case ID_COPY_RST_SENT:
	case ID_COPY_RST_RCVD:
	case ID_COPY_NOTES:
	{
		SendMessage(hWnd, EM_SETSEL, nStartChar, nEndChar);
		SendMessage(hWnd, WM_COPY, 0, 0);
		OpenClipboard(NULL);
		hClipboardData = GetClipboardData(
#ifdef _WIN32_WCE
		CF_UNICODETEXT
#else /* _WIN32_WCE */
		CF_TEXT
#endif /* _WIN32_WCE */
		);
		if (hClipboardData != 0) {
			LPTSTR pString = (LPTSTR)GlobalLock(hClipboardData);
			int    nLen    = _tcslen(pString);
			int    iLine   = iCmd - ID_COPY_CALL;
			QsoInfo *info  = trx->pQso;
			GlobalUnlock(hClipboardData);
			QSOINFO_ARRAY[iLine] = (TCHAR*)realloc(
				QSOINFO_ARRAY[iLine], (nLen + 1) * sizeof(TCHAR));
			memcpy(QSOINFO_ARRAY[iLine], pString, (nLen + 1) * sizeof(TCHAR));
		}
		CloseClipboard();
		break;
	}
	default:
		break;
	}
}

LRESULT CALLBACK rx_wndproc(HWND hWnd, UINT message, WPARAM wParam, LPARAM lParam)
{
#ifdef _WIN32_WCE
	if (message == WM_LBUTTONDOWN) {
		SHRGINFO shrg;
		shrg.cbSize		= sizeof(SHRGINFO);
		shrg.dwFlags	= SHRG_RETURNCMD;
		shrg.hwndClient = hWnd;
		shrg.ptDown.x	= LOWORD(lParam);
		shrg.ptDown.y	= HIWORD(lParam);
		if (SHRecognizeGesture(&shrg) == GN_CONTEXTMENU) {
			ShowContextMenu(hWnd, shrg.ptDown.x, shrg.ptDown.y, FALSE);
			return 0;
		}
	} else 
#endif /* _WIN32_WCE */
	if (message == WM_RBUTTONDOWN) {
		ShowContextMenu(hWnd, LOWORD(lParam), HIWORD(lParam), FALSE);
		return 0;
	}
	return CallWindowProc(edit_wndproc, hWnd, message, wParam, lParam);
}

LRESULT CALLBACK tx_wndproc(HWND hWnd, UINT message, WPARAM wParam, LPARAM lParam)
{
#ifdef _WIN32_WCE
	if (message == WM_LBUTTONDOWN) {
		SHRGINFO shrg;
		shrg.cbSize		= sizeof(SHRGINFO);
		shrg.dwFlags	= SHRG_RETURNCMD;
		shrg.hwndClient = hWnd;
		shrg.ptDown.x	= LOWORD(lParam);
		shrg.ptDown.y	= HIWORD(lParam);
		if (SHRecognizeGesture(&shrg) == GN_CONTEXTMENU) {
			ShowContextMenu(hWnd, shrg.ptDown.x, shrg.ptDown.y, TRUE);
			return 0;
		}
	}
	else
#endif /* _WIN32_WCE */
	if (message == WM_RBUTTONDOWN) {
		ShowContextMenu(hWnd, LOWORD(lParam), HIWORD(lParam), TRUE);
		return 0;
	}
	return CallWindowProc(edit_wndproc, hWnd, message, wParam, lParam);
}

/* ---------------------------------------------------------------------- */

//FIXME
/*
static pthread_mutex_t trx_mutex = PTHREAD_MUTEX_INITIALIZER;
static pthread_cond_t trx_cond = PTHREAD_COND_INITIALIZER;
static pthread_t trx_thread;
*/

/* ---------------------------------------------------------------------- */

#define BLOCKLEN	512

#if 0
static void receive_loop(void)
{
	float *buf;
	int len;

	if (sound_open_for_read(trx.samplerate) < 0) {
		errmsg("sound_open_for_read: %s", sound_error());
		trx_set_state(TRX_STATE_ABORT);
		return;
	}

	trx.rxinit(&trx);
	statusbar_set_trxstate(TRX_STATE_RX);

	for (;;) {
		pthread_mutex_lock(&trx_mutex);

		if (trx.state != TRX_STATE_RX) {
			pthread_mutex_unlock(&trx_mutex);
			break;
		}

		pthread_mutex_unlock(&trx_mutex);

		len = BLOCKLEN;

		if (sound_read(&buf, &len) < 0) {
			errmsg("%s", sound_error());
			trx_set_state(TRX_STATE_ABORT);
			break;
		}

		waterfall_set_data(waterfall, buf, len);
		trx.rxprocess(&trx, buf, len);
	}

	sound_close();
}

static void transmit_loop(void)
{
	float f = 0.0;

	if (sound_open_for_write(trx.samplerate) < 0) {
		errmsg("sound_open_for_write: %s", sound_error());
		trx_set_state(TRX_STATE_ABORT);
		return;
	}

	trx_set_scope(&f, 1, FALSE);
	trx_set_phase(0.0, FALSE);
//	trx.metric = 0.0;

	trx.txinit(&trx);
	ptt_set(1);

	for (;;) {
		pthread_mutex_lock(&trx_mutex);

		if (trx.state == TRX_STATE_ABORT) {
			pthread_mutex_unlock(&trx_mutex);
			break;
		}

		if (trx.state == TRX_STATE_RX || trx.state == TRX_STATE_PAUSE) {
			statusbar_set_trxstate(TRX_STATE_FLUSH);
			trx.stopflag = 1;
		}

		if (trx.state == TRX_STATE_TUNE) {
			statusbar_set_trxstate(TRX_STATE_TUNE);
			trx.tune = 1;
		}

		if (trx.state == TRX_STATE_TX) {
			statusbar_set_trxstate(TRX_STATE_TX);
		}

		pthread_mutex_unlock(&trx_mutex);

		if (trx.txprocess(&trx) < 0)
			break;
	}

	sound_close();
	ptt_set(0);

	pthread_mutex_lock(&trx_mutex);
	g_free(trx.txstr);
	trx.txptr = NULL;
	trx.txstr = NULL;
	trx.backspaces = 0;
	trx.stopflag = 0;
	trx.tune = 0;
	pthread_mutex_unlock(&trx_mutex);

	clear_tx_text();
}

static void *trx_loop(void *args)
{
	for (;;) {
		pthread_mutex_lock(&trx_mutex);

		if (trx.state == TRX_STATE_PAUSE || \
		    trx.state == TRX_STATE_ABORT)
			break;

		/* signal a state change (other than pause or abort) */
		pthread_cond_signal(&trx_cond);

		switch (trx.state) {
		case TRX_STATE_RX:
			pthread_mutex_unlock(&trx_mutex);
			receive_loop();
			break;

		case TRX_STATE_TX:
		case TRX_STATE_TUNE:
			pthread_mutex_unlock(&trx_mutex);
			transmit_loop();
			break;

		default:
			pthread_mutex_unlock(&trx_mutex);
			break;
		}
	}

	trx.destructor(&trx);

	trx.modem = NULL;
	trx.state = TRX_STATE_PAUSE;
//	trx.metric = 0.0;

	g_free(trx.txstr);
	trx.txstr = NULL;
	trx.txptr = NULL;

	/* do this before sending the signal, otherwise there is a race */
	statusbar_set_trxstate(TRX_STATE_PAUSE);

	/* signal a state change */
	pthread_cond_signal(&trx_cond);

	pthread_mutex_unlock(&trx_mutex);

	/* this will exit the trx thread */
	return NULL;
}

/* ---------------------------------------------------------------------- */

int trx_init(void)
{
	pthread_mutex_lock(&trx_mutex);

	if (trx.modem) {
		pthread_mutex_unlock(&trx_mutex);
		return -1;
	}

	switch (trx.mode) {
	case MODE_MFSK16:
		mfsk_init(&trx);
		break;

	case MODE_MFSK8:
		mfsk_init(&trx);
		break;

	case MODE_RTTY:
		rtty_init(&trx);
		break;

	case MODE_THROB1:
	case MODE_THROB2:
	case MODE_THROB4:
		throb_init(&trx);
		break;

	case MODE_BPSK31:
	case MODE_QPSK31:
	case MODE_PSK63:
		psk31_init(&trx);
		break;

	case MODE_MT63:
		mt63_init(&trx);
		break;

	case MODE_FELDHELL:
	case MODE_FMHELL:
		feld_init(&trx);
		break;
	}

	if (trx.modem == NULL) {
		errmsg("Modem initialization failed!");
		pthread_mutex_unlock(&trx_mutex);
		return -1;
	}

	trx.stopflag = 0;

	pthread_mutex_unlock(&trx_mutex);

	if (pthread_create(&trx_thread, NULL, trx_loop, NULL) < 0) {
		errmsg("pthread_create: %m");
		trx.state = TRX_STATE_PAUSE;
		return -1;
	}

	return 0;
}

/* ---------------------------------------------------------------------- */

void trx_set_mode(trx_mode_t mode)
{
	trx.mode = mode;
}

trx_mode_t trx_get_mode(void)
{
	return trx.mode;
}

char *trx_get_mode_name(void)
{
	return trx_mode_names[trx.mode];
}

void trx_set_state(trx_state_t state)
{
#if 0
	fprintf(stderr,
		"trx_set_state: old=%s new=%s\n", 
		trx_state_names[trx.state], 
		trx_state_names[state]);
#endif

	pthread_mutex_lock(&trx_mutex);

	if (trx.state == state) {
		pthread_mutex_unlock(&trx_mutex);
		return;
	}
	trx.state = state;

	pthread_mutex_unlock(&trx_mutex);

	if (state == TRX_STATE_PAUSE || state == TRX_STATE_ABORT)
		return;

	/* (re-)initialize the trx */
	trx_init();
}

void trx_set_state_wait(trx_state_t state)
{
#if 0
	fprintf(stderr,
		"trx_set_state_wait: old=%s new=%s\n", 
		trx_state_names[trx.state], 
		trx_state_names[state]);
#endif

	pthread_mutex_lock(&trx_mutex);

	if (trx.state == state) {
		pthread_mutex_unlock(&trx_mutex);
		return;
	}

	trx.state = state;

	if (trx.modem) {
		/* now wait for the main trx loop to respond */
		pthread_cond_wait(&trx_cond, &trx_mutex);
	}

	pthread_mutex_unlock(&trx_mutex);

	if (state == TRX_STATE_PAUSE || state == TRX_STATE_ABORT)
		return;

	/* (re-)initialize the trx */
	trx_init();
}
#endif /* 0 */

void trx_set_rtty_parms(Trx *trx, float squelch, int shift, float baud,
			int bits, int parity, int stop,
			BOOL reverse, BOOL msbfirst)
{
	trx->rtty_squelch = squelch;
	trx->rtty_shift = shift;
	trx->rtty_baud = baud;
	trx->rtty_bits = bits;
	trx->rtty_stop = stop;
	trx->rtty_reverse = reverse;
	trx->rtty_msbfirst = msbfirst;
}

/*

trx_state_t trx_get_state(void)
{
	return trx.state;
}

void trx_set_freq(float freq)
{
	trx.frequency = freq;
}

float trx_get_freq(void)
{
	return trx.frequency;
}

float trx_get_bandwidth(void)
{
	return trx.bandwidth;
}

void trx_set_squelch(BOOL on)
{
	trx.squelchon = on;
}

void trx_set_reverse(BOOL on)
{
	trx.reverse = on;
}
*/

#if 0
void trx_set_metric(float metric)
{
	trx.metric = metric;
}

float trx_get_metric(void)
{
	return trx.metric;
}

void trx_set_txoffset(float txoffset)
{
	trx.txoffset = txoffset;
}

float trx_get_txoffset(void)
{
	return trx.txoffset;
}

void trx_set_mfsk_parms(float squelch)
{
	trx.mfsk_squelch = squelch;
}

void trx_set_throb_parms(float squelch)
{
	trx.throb_squelch = squelch;
}

void trx_set_mt63_parms(float squelch,
			int bandwidth, int interleave,
			BOOL cwid, BOOL esc)
{
	trx.mt63_squelch = squelch;
	trx.mt63_bandwidth = bandwidth;
	trx.mt63_interleave = interleave;
	trx.mt63_cwid = cwid;
	trx.mt63_esc = esc;
}

void trx_set_hell_parms(gchar *font,
			BOOL upper,
			float bandwidth, float agcattack, float agcdecay)
{
	trx.hell_font = font;
	trx.hell_upper = upper;
	trx.hell_bandwidth = bandwidth;
	trx.hell_agcattack = agcattack;
	trx.hell_agcdecay = agcdecay;
}

#endif /* 0 */

/*
void trx_set_scope(int *data, int len, BOOL autoscale)
{
//FIXME
//	Miniscope *m = MINISCOPE(lookup_widget(appwindow, "miniscope"));
//	miniscope_set_data(m, data, len, autoscale);
}
*/

void trx_set_phase(int phase, BOOL highlight)
{
//FIXME
//	Miniscope *m = MINISCOPE(lookup_widget(appwindow, "miniscope"));
//	miniscope_set_phase(m, phase, highlight);
}

#if 0
int trx_get_samplerate(void)
{
	return trx.samplerate;
}

/* ---------------------------------------------------------------------- */

static GAsyncQueue *rx_queue 			= NULL;
static GAsyncQueue *rx_hell_data_queue 		= NULL;
static GAsyncQueue *rx_picture_data_queue 	= NULL;
static GAsyncQueue *echo_queue 			= NULL;
static GAsyncQueue *tx_picture_queue 		= NULL;

void trx_init_queues(void)
{
	rx_queue 		= g_async_queue_new();
	rx_hell_data_queue 	= g_async_queue_new();
	rx_picture_data_queue 	= g_async_queue_new();
	echo_queue 		= g_async_queue_new();
	tx_picture_queue 	= g_async_queue_new();
}

/*
 * A very ugly hack. You can not g_async_queue_push() a 0 ...
 */
static inline gpointer gint_to_pointer(int data)
{
	return (data == 0) ? GINT_TO_POINTER(-1) : GINT_TO_POINTER(data);
}

static inline int gpointer_to_int(gpointer data)
{
	return (GPOINTER_TO_INT(data) == -1) ? 0 : GPOINTER_TO_INT(data);
}

#endif /* 0 */

#if 0
int trx_get_rx_char(void)
{
	gpointer data;

	g_return_val_if_fail(rx_queue, -1);
	data = g_async_queue_try_pop(rx_queue);
	return data ? gpointer_to_int(data) : -1;
}

void trx_put_rx_data(guint data)
{
	g_return_if_fail(rx_hell_data_queue);
	g_async_queue_push(rx_hell_data_queue, gint_to_pointer(data));
}

int trx_get_rx_data(void)
{
	gpointer data;

	g_return_val_if_fail(rx_hell_data_queue, -1);
	data = g_async_queue_try_pop(rx_hell_data_queue);
	return data ? gpointer_to_int(data) : -1;
}

void trx_put_rx_picdata(guint data)
{
	g_return_if_fail(rx_picture_data_queue);
	g_async_queue_push(rx_picture_data_queue, gint_to_pointer(data));
}

int trx_get_rx_picdata(void)
{
	gpointer data;

	g_return_val_if_fail(rx_picture_data_queue, -1);
	data = g_async_queue_try_pop(rx_picture_data_queue);
	return data ? gpointer_to_int(data) : -1;
}
#endif /* 0 */

#if 0
int trx_get_echo_char(void)
{
	gpointer data;

	g_return_val_if_fail(echo_queue, -1);
	data = g_async_queue_try_pop(echo_queue);
	return data ? gpointer_to_int(data) : -1;
}

void trx_put_tx_picture(gpointer picbuf)
{
	g_return_if_fail(tx_picture_queue);
	g_return_if_fail(picbuf);
	g_async_queue_push(tx_picture_queue, picbuf);
}

gpointer trx_get_tx_picture(void)
{
	g_return_val_if_fail(tx_picture_queue, NULL);
	return g_async_queue_try_pop(tx_picture_queue);
}

/* ---------------------------------------------------------------------- */

void trx_queue_backspace(void)
{
	trx.backspaces++;
}

static BOOL check_for_spaces(gchar *str)
{
	gunichar chr;

	while (*str) {
		chr = g_utf8_get_char(str);

		if (g_unichar_isspace(chr) || chr == TRX_RX_CMD)
			return TRUE;

		str = g_utf8_next_char(str);
	}

	return FALSE;
}

gunichar trx_get_hell_tx_char(void)
{
	GtkEditable *editable;
	gunichar chr;
	gchar *str;
	int pos;
	BOOL spc;

	gdk_threads_enter();

	editable = GTK_EDITABLE(lookup_widget(appwindow, "txentry"));
	str = gtk_editable_get_chars(editable, 0, -1);

	chr = g_utf8_get_char(str);
	spc = check_for_spaces(str);

	g_free(str);

	/*
	 * If there are no spaces in the text, then send idle.
	 */
	if (chr == 0 || spc == FALSE) {
		gdk_threads_leave();
		return -1;
	}

	/*
	 * Delete the first character and move cursor one step
	 * backwards.
	 */
	pos = gtk_editable_get_position(editable);
	gtk_editable_delete_text(editable, 0, 1);
	gtk_editable_set_position(editable, pos - 1);

	if (chr == TRX_RX_CMD) {
		push_button("rxbutton");
		chr = -1;
	}

	gdk_threads_leave();

	return chr;
}

gunichar trx_get_normal_tx_char(void)
{
	GtkTextView *view;
	GtkTextBuffer *buffer;
	GtkTextIter iter1, iter2;
	GtkTextMark *mark1, *mark2;
	gunichar chr;

	gdk_threads_enter();

	view = GTK_TEXT_VIEW(lookup_widget(appwindow, "txtext"));
	buffer = gtk_text_view_get_buffer(view);

	/* get the iterator at TX mark */
	mark1 = gtk_text_buffer_get_mark(buffer, "editablemark");
	gtk_text_buffer_get_iter_at_mark(buffer, &iter1, mark1);

	/* get the iterator at the insert mark */
	mark2 = gtk_text_buffer_get_insert(buffer);
	gtk_text_buffer_get_iter_at_mark(buffer, &iter2, mark2);

	if (gtk_text_iter_equal(&iter1, &iter2)) {
		gdk_threads_leave();
		return -1;
	}

	chr = gtk_text_iter_get_char(&iter1);

	if (chr == 0) {
		gdk_threads_leave();
		return -1;
	}

	gtk_text_iter_forward_char(&iter1);
	gtk_text_buffer_get_start_iter(buffer, &iter2);
	gtk_text_buffer_apply_tag_by_name(buffer, "editabletag",
					  &iter2, &iter1);
	gtk_text_buffer_move_mark_by_name(buffer, "editablemark", &iter1);

	if (chr == TRX_RX_CMD) {
		push_button("rxbutton");
		chr = -1;
	}

	gdk_threads_leave();

	return chr;
}

gunichar trx_get_tx_char(void)
{
        GError *err = NULL;
	guchar utf[8], *str;
	gunichar chr;
	int len;
	gchar *to = "ISO-8859-1";
	gchar *fm = "UTF-8";
	gchar *fb = ".";

	if (trx.mode == MODE_FELDHELL || trx.mode == MODE_FMHELL)
		return trx_get_hell_tx_char();

	if (trx.txstr && trx.txptr && *trx.txptr)
		return *trx.txptr++;

	g_free(trx.txstr);
	trx.txptr = NULL;
	trx.txstr = NULL;

	if (trx.backspaces > 0) {
		trx.backspaces--;
		return 8;
	}

	chr = trx_get_normal_tx_char();

	if (chr == -1 || 0xFFFC)
		return chr;

	len = g_unichar_to_utf8(chr, utf);
	str = g_convert_with_fallback(utf, len, to, fm, fb, NULL, NULL, &err);

	if (!str) {
		if (err) {
			g_warning("trx_get_tx_char: conversion failed: %s", err->message);
			g_error_free(err);
		}
		return -1;
	}

	trx.txptr = trx.txstr = str;

	return *trx.txptr++;
}

#endif /* 0 */
