#include "sliderpopup.h"

#include <windows.h>
#include <commctrl.h>
#include <memory.h>
#include <stdio.h>
#include <tchar.h>

#include "../Resource.h"

/// Forward reference to the sliderpopup window procedure
LRESULT CALLBACK sliderpopup_wndproc(HWND, UINT, WPARAM, LPARAM);

const TBYTE WIN32CLASS_SLIDERPOPUP[] = _T("SLIDERPOPUP");

/// Register class of the waterfall window
ATOM sliderpopup_register_class(HINSTANCE hInstance)
{
	WNDCLASS	wc;

	wc.style			= CS_HREDRAW | CS_VREDRAW;
	wc.lpfnWndProc		= (WNDPROC) sliderpopup_wndproc;
	wc.cbClsExtra		= 0;
	wc.cbWndExtra		= 0;
	wc.hInstance		= hInstance;
	wc.hIcon			= 0;
	wc.hCursor			= 0;
	wc.hbrBackground	= (HBRUSH)COLOR_BTNFACE;
	wc.lpszMenuName		= 0;
	wc.lpszClassName	= WIN32CLASS_SLIDERPOPUP;

	return RegisterClass(&wc);
}

SliderPopup* sliderpopup_new()
{
	SliderPopup *sp = (SliderPopup*)malloc(sizeof(SliderPopup));
	if (sp == 0)
		return 0;
	if (! sliderpopup_init(sp)) {
		sliderpopup_destroy(sp);
		return 0;
	}
	return sp;
}

void sliderpopup_free(SliderPopup *sp)
{
	if (sp == 0)
		return;
	sliderpopup_destroy(sp);
	free(sp);
}

BOOL sliderpopup_init(SliderPopup *sp)
{
	memset(sp, 0, sizeof(SliderPopup));
	return TRUE;
}

BOOL sliderpopup_create(SliderPopup *sp, DWORD dwStyle,
	int x, int y, int nWidth, int nHeight, 
	LPCTSTR pTitle, LPCTSTR pFormat, BOOL bHasSwitch, UINT iCmd, HWND hWndParent, HANDLE hInstance)
{
//	HRGN  hrgnWindow = CreateRoundRectRgn(0, 0, nWidth, nHeight, 12, 12);
	HFONT hFont		 = (HFONT)SendMessage(hWndParent, WM_GETFONT, 0, 0);
	sp->hWnd = CreateWindow(WIN32CLASS_SLIDERPOPUP, NULL, dwStyle,
		0, 0, 0, 0, hWndParent, NULL, hInstance, (void*)sp);
	if (! sp->hWnd)
		return FALSE;
	SetWindowLong(sp->hWnd, GWL_USERDATA, (long)sp);

	sp->pTitle				 = _tcsdup(pTitle);
	sp->pFormat				 = _tcsdup(pFormat);
	sp->iOnCloseNotification = iCmd;
	sp->bUpdateOnDragging    = FALSE;

//	SetWindowRgn(sp->hWnd, hrgnWindow, FALSE);

	{
		LOGFONT logfont;
		memset(&logfont, 0, sizeof(LOGFONT));
		logfont.lfHeight  = -12;
		logfont.lfWeight  = FW_NORMAL;
		logfont.lfCharSet = ANSI_CHARSET;
		_tcscpy(logfont.lfFaceName, _T("Tahoma"));
		sp->hFont = CreateFontIndirect(&logfont);
	}

	SendMessage(sp->hWnd, WM_SETFONT, (WPARAM)sp->hFont, FALSE);

	sp->hWndSlider = CreateWindow(_T("msctls_trackbar32"), _T(""), TBS_VERT | WS_TABSTOP | WS_CHILD | WS_CLIPSIBLINGS | WS_VISIBLE, 
		10, 30, 40, nHeight - (bHasSwitch ? 84 : 60), sp->hWnd, 0, hInstance, 0);
	sp->hWndValue  = CreateWindow(_T("STATIC"), _T("30 WPM"), WS_CHILD | WS_CLIPSIBLINGS | WS_VISIBLE,
		10, nHeight - (bHasSwitch ? 47 : 23), nWidth - 20, 20, sp->hWnd, 0, hInstance, 0);
	if (bHasSwitch) {
		sp->hWndSwitch = CreateWindow(_T("BUTTON"), _T("Enabled"), WS_CHILD | WS_CLIPSIBLINGS | WS_VISIBLE | BS_AUTOCHECKBOX,
			10, nHeight - 25, nWidth - 13, 20, sp->hWnd, 0, hInstance, 0);
	}

	SendMessage(sp->hWndSlider, WM_SETFONT, (WPARAM)hFont, FALSE);
	SendMessage(sp->hWndValue,  WM_SETFONT, (WPARAM)sp->hFont, FALSE);
	SendMessage(sp->hWndSwitch, WM_SETFONT, (WPARAM)sp->hFont, FALSE);

	SetWindowPos(sp->hWnd, HWND_TOP, x, y, nWidth, nHeight, SWP_NOOWNERZORDER | SWP_SHOWWINDOW);
	SetFocus(sp->hWndSlider);
	sp->bActive = TRUE;

	return TRUE;
}

void sliderpopup_destroy(SliderPopup *sp)
{
	SetWindowLong(sp->hWnd, GWL_USERDATA, 0);
	DeleteObject(sp->hFont);
	DestroyWindow(sp->hWndSlider);
	DestroyWindow(sp->hWndValue);
	DestroyWindow(sp->hWndSwitch);
	DestroyWindow(sp->hWnd);
	free(sp->pTitle);
	free(sp->pFormat);
	sp->pTitle		= NULL;
	sp->pFormat		= NULL;
	sp->hWnd		= NULL;
	sp->hWndSlider	= NULL;
	sp->hWndValue	= NULL;
	sp->hWndSwitch	= NULL;
	sp->hFont		= NULL;
}

BOOL sliderpopup_pretranslate_message(SliderPopup *sp, MSG *msg)
{
	BOOL bClose = FALSE;

	if (! sp->bActive)
		return FALSE;

	if (msg->message == WM_LBUTTONDOWN		|| msg->message == WM_RBUTTONDOWN || 
		msg->message == WM_MBUTTONDOWN		|| msg->message == WM_LBUTTONDBLCLK || 
		msg->message == WM_RBUTTONDBLCLK	|| msg->message == WM_MBUTTONDBLCLK
#ifndef _WIN32_WCE
		||
		msg->message == WM_NCLBUTTONDOWN	|| msg->message == WM_NCRBUTTONDOWN ||
		msg->message == WM_NCMBUTTONDOWN	|| msg->message == WM_NCLBUTTONDBLCLK || 
		msg->message == WM_NCRBUTTONDBLCLK	|| msg->message == WM_NCMBUTTONDBLCLK
#endif /* _WIN32_WCE */
		) {
		// test, whehther the click is outside the popup
		RECT  rect;
		POINT pt;
		pt.x = LOWORD(msg->lParam); 
		pt.y = HIWORD(msg->lParam);
		ClientToScreen(msg->hwnd, &pt);
		GetWindowRect(sp->hWnd, &rect);
		bClose = ! PtInRect(&rect, pt);
	}

	if (msg->message == WM_SETFOCUS) {
		HWND hWnd = (HWND)msg->wParam;
		if (hWnd == 0 || (hWnd != sp->hWnd && hWnd != sp->hWndSlider))
			bClose = TRUE;
	}

	if (msg->message == WM_ACTIVATE
#ifndef _WIN32_WCE
		|| msg->message == WM_ACTIVATEAPP || msg->message == WM_NCACTIVATE
#endif /* _WIN32_WCE */
		)
		bClose = TRUE;

	if (bClose) {
		sp->bActive = 0;
		PostMessage(GetParent(sp->hWnd), WM_COMMAND, sp->iOnCloseNotification, 0);
		sliderpopup_destroy(sp);
	}

	return FALSE;
}

int sliderpopup_reverse_value(SliderPopup *sp, int iValue)
{
	int iMin = SendMessage(sp->hWndSlider, TBM_GETRANGEMIN, 0, 0);
	int iMax = SendMessage(sp->hWndSlider, TBM_GETRANGEMAX, 0, 0);
	return iMax - (iValue - iMin);
}

void sliderpopup_set_value(SliderPopup *sp, int iValue)
{
	TCHAR buf[256];
	int iValueReversed = sliderpopup_reverse_value(sp, iValue);
	sp->iValue = iValue;
	if (iValueReversed != SendMessage(sp->hWndSlider, TBM_GETPOS, 0, 0))
		SendMessage(sp->hWndSlider, TBM_SETPOS, TRUE, iValueReversed);
	_stprintf(buf, sp->pFormat, sp->iValue);
	SetWindowText(sp->hWndValue, buf);
	if (sp->bUpdateOnDragging)
		PostMessage(GetParent(sp->hWnd), WM_COMMAND, sp->iOnCloseNotification, 0);
}

void sliderpopup_set_switch(SliderPopup *sp, BOOL bSwitch)
{
	sp->bSwitchValue = bSwitch;
	SendMessage(sp->hWndSwitch, BM_SETCHECK, bSwitch ? BST_CHECKED : BST_UNCHECKED, 0);
	if (sp->bUpdateOnDragging)
		PostMessage(GetParent(sp->hWnd), WM_COMMAND, sp->iOnCloseNotification, 0);
}

LRESULT CALLBACK sliderpopup_wndproc(HWND hWnd, UINT message, WPARAM wParam, LPARAM lParam)
{
	SliderPopup *sp = (SliderPopup*)GetWindowLong(hWnd, GWL_USERDATA);
	if (sp == 0)
		return DefWindowProc(hWnd, message, wParam, lParam);

	if (message == WM_PAINT) {
		int nRounding = 12;
#ifdef _WIN32_WCE
		COLORREF cTitle	     = RGB(255, 255, 255);
		COLORREF cCaption    = RGB(0, 0, 128);
		COLORREF cBackground = RGB(255, 255, 255);
#else  /* _WIN32_WCE */
		COLORREF cTitle	     = GetSysColor(COLOR_CAPTIONTEXT);
		COLORREF cCaption    = GetSysColor(COLOR_ACTIVECAPTION);
		COLORREF cBackground = GetSysColor(COLOR_BTNFACE);
#endif /* _WIN32_WCE */
//		HRGN hrgn = CreateRectRgn(0,0,0,0);

		PAINTSTRUCT paint;
		RECT rect;
		HBRUSH	hBrushTitle, hBrushWindow, hBrushOld;
		HPEN    hPenTitle, hPenBorder, hPenOld;
		BeginPaint(hWnd, &paint);
		
		hBrushWindow = CreateSolidBrush(cBackground);
//		hBrushOld    = SelectObject(paint.hdc, hBrushWindow);
		FillRect(paint.hdc, &paint.rcPaint, hBrushWindow);
//		SelectObject(hBrushOld);
		DeleteObject(hBrushWindow);

		memcpy(&rect, &paint.rcPaint, sizeof(RECT));
		rect.top    = 2;
		rect.bottom = rect.top + 23;
		hBrushTitle = CreateSolidBrush(cCaption);
		FillRect(paint.hdc, &rect, hBrushTitle);
		DeleteObject(hBrushTitle);

//		hBrushTitle  = CreateSolidBrush(RGB(0, 0, 0));
//		GetWindowRgn(hWnd, hrgn);
//		OffsetRgn(hrgn, -2, -2);
//		FrameRgn(paint.hdc, hrgn, (HBRUSH)GetStockObject(BLACK_BRUSH), 2, 2);
//		DeleteObject(hrgn);

		{
			RECT rect;
//			HRGN hRgn;
			POINT aPoints[5];
			GetWindowRect(sp->hWndSlider, &rect);
			ScreenToClient(sp->hWnd, (LPPOINT)&rect.left);
			ScreenToClient(sp->hWnd, (LPPOINT)&rect.right);
			aPoints[0].x = rect.right + 10;
			aPoints[0].y = rect.bottom;
			aPoints[1].x = rect.right + 25;
			aPoints[1].y = rect.top;
			aPoints[2].x = aPoints[0].x;
			aPoints[2].y = aPoints[1].y;
			SelectBrush(paint.hdc, GetStockObject(BLACK_BRUSH));
			Polygon(paint.hdc, aPoints, 3); 

			aPoints[0].x = paint.rcPaint.left;
			aPoints[0].y = paint.rcPaint.top;
			aPoints[1].x = paint.rcPaint.left;
			aPoints[1].y = paint.rcPaint.bottom - 1;
			aPoints[2].x = paint.rcPaint.right - 1;
			aPoints[2].y = paint.rcPaint.bottom - 1;
			aPoints[3].x = paint.rcPaint.right - 1;
			aPoints[3].y = paint.rcPaint.top;
			aPoints[4].x = paint.rcPaint.left;
			aPoints[4].y = paint.rcPaint.top;
			Polyline(paint.hdc, aPoints, 5); 
			aPoints[0].x = paint.rcPaint.left + 1;
			aPoints[0].y = paint.rcPaint.top + 1;
			aPoints[1].x = paint.rcPaint.left + 1;
			aPoints[1].y = paint.rcPaint.bottom - 2;
			aPoints[2].x = paint.rcPaint.right - 2;
			aPoints[2].y = paint.rcPaint.bottom - 2;
			aPoints[3].x = paint.rcPaint.right - 2;
			aPoints[3].y = paint.rcPaint.top + 1;
			aPoints[4].x = paint.rcPaint.left + 1;
			aPoints[4].y = paint.rcPaint.top + 1;
			Polyline(paint.hdc, aPoints, 5); 
			
//			FillRgn(paint.hdc, hRgn, (HBRUSH)GetStockObject(BLACK_BRUSH));
//			DeleteObject(hRgn);
		}

		SetTextColor(paint.hdc, cTitle);
		SetBkColor(paint.hdc, cCaption);

//		hBrushTitle  = CreateSolidBrush(GetSysColor(COLOR_ACTIVECAPTION));
//		hBrushOld    = SelectObject(paint.hdc, (HBRUSH)(COLOR_ACTIVECAPTION+1));
//		hPenTitle    = CreatePen(PS_SOLID, 0, GetSysColor(COLOR_ACTIVECAPTION));
//		hPenOld      = SelectObject(paint.hdc, hPenTitle);
		{
			HFONT hFontOld = SelectFont(paint.hdc, sp->hFont);
			rect.left += 5;
			DrawText(paint.hdc, sp->pTitle, _tcslen(sp->pTitle), &rect, DT_LEFT | DT_NOPREFIX | DT_SINGLELINE | DT_VCENTER);
			SelectFont(paint.hdc, hFontOld);
		}
//		ExtTextOut(paint.hdc, 8, 3, 0, 0, _T("Ahoj"), 4, 0);
//		SelectObject(paint.hdc, hPenOld);
//		SelectObject(paint.hdc, hBrushOld);

/*
		hPenTitle    = CreatePen(PS_SOLID, 0, GetSysColor(COLOR_ACTIVECAPTION));
		hPenBorder   = CreatePen(PS_SOLID, 0, RGB(0, 0, 0)); // GetSysColor(COLOR_ACTIVEBORDER));
		hBrushTitle  = CreateSolidBrush(GetSysColor(COLOR_ACTIVECAPTION));
		hBrushWindow = CreateSolidBrush(GetSysColor(COLOR_STATIC));
		hPenOld      = SelectObject(paint.hdc, hPenTitle);
		hBrushOld    = SelectObject(paint.hdc, hBrushTitle);
		FillRect(paint.hdc, 
			paint.rcPaint.left, paint.rcPaint.top, paint.rcPaint.right, paint.rcPaint.top + 20);
		SelectObject(paint.hdc, hBrushTitle);
		FillRect(paint.hdc,
			paint.rcPaint.left, paint.rcPaint.top, paint.rcPaint.right, paint.rcPaint.bottom);
//		draw(sp, BeginPaint(hWnd, &paint), TRUE);
		SelectObject(paint.hdc, hPenOld);
		SelectObject(paint.hdc, hBrushOld);
*/
		EndPaint(hWnd, &paint);
	} else if (message == WM_ERASEBKGND) {
		// ignore background erasing
		return TRUE;
	} else if (message == WM_LBUTTONDOWN) {
	} else if (message == WM_MOUSEMOVE) {
	} else if (message == WM_LBUTTONUP) {
	} else if (message == WM_DESTROY) {
		sliderpopup_destroy(sp);
	} else if (message == WM_VSCROLL) {
		sliderpopup_set_value(sp, sliderpopup_reverse_value(
			sp, SendMessage(sp->hWndSlider, TBM_GETPOS, 0, 0l)));
		return 0;
	} else if (message == WM_COMMAND) {
		sliderpopup_set_switch(sp, SendMessage(sp->hWndSwitch, BM_GETCHECK, 0, 0l) == BST_CHECKED);
		return 0;
	}

	return DefWindowProc(hWnd, message, wParam, lParam);
}
