#include <windows.h>

#ifdef _WIN32_WCE
#define _ASSERT(co)
#else
#include <crtdbg.h>
#endif /* _WIN32_WCE */

#include "ColorTable.h"

COLORREF HSV_2_RGB(double H, double S, double L);

void CreateColorGradient(
	double dHue0, double dHue1, double dSat0, double dSat1, double dVal0, double dVal1,
	COLORREF *pColorsOut, int nColorsOut)
{
	int i;
//	pColorsOut[0] = 0;
	for (i = 0; i < nColorsOut; ++ i) {
		double dPwr = (double)i/(double)(nColorsOut - 1);
		double dHue =  dHue0 + dPwr * (dHue1 - dHue0);
		double dSat =  dSat0 + dPwr * (dSat1 - dSat0);
		double dVal =  dVal0 + dPwr * (dVal1 - dVal0);
		pColorsOut[i] = HSV_2_RGB(dHue, dSat, dVal);
	}
}

void CreatePowerColorGradient16bit(COLORREF *pColorsOut, int nColorsOut)
{
	int i;
	int iColor = 0;

	_ASSERT(nColorsOut == 91);

	for (i = 0; i < 16; ++ i)
		pColorsOut[iColor ++] = RGB(0, 0, i << 4);
	for (i = 1; i < 16; ++ i)
		pColorsOut[iColor ++] = RGB(0, i << 4, 15 << 4);
	for (i = 1; i < 16; ++ i)
		pColorsOut[iColor ++] = RGB(0, 15 << 4, (16 - i) << 4);
	for (i = 1; i < 16; ++ i)
		pColorsOut[iColor ++] = RGB(i << 4, 15 << 4, 0);
	for (i = 1; i < 16; ++ i)
		pColorsOut[iColor ++] = RGB(15 << 4, (16 - i) << 4, 0);
	for (i = 1; i < 16; ++ i)
		pColorsOut[iColor ++] = RGB(15 << 4, 0, i << 4);
}

void CreatePowerColorGradientSimple(int nBits, COLORREF *pColorsOut, int nColorsOut)
{
	int i;
	int iColor  = 0;
	int iShift  = 8 - nBits;
	int iMaxVal = 1;

	for (i = 0; i < nBits; ++ i)
		iMaxVal <<= 1;
	-- iMaxVal;

	_ASSERT(nColorsOut == iMaxVal * 6 + 1);

	for (i = 0; i <= iMaxVal; ++ i)
		pColorsOut[iColor ++] = RGB(0, 0, i << iShift);
	for (i = 1; i <= iMaxVal; ++ i)
		pColorsOut[iColor ++] = RGB(0, i << iShift, iMaxVal << iShift);
	for (i = 1; i <= iMaxVal; ++ i)
		pColorsOut[iColor ++] = RGB(0, iMaxVal << iShift, (iMaxVal + 1 - i) << iShift);
	for (i = 1; i <= iMaxVal; ++ i)
		pColorsOut[iColor ++] = RGB(i << iShift, iMaxVal << iShift, 0);
	for (i = 1; i <= iMaxVal; ++ i)
		pColorsOut[iColor ++] = RGB(iMaxVal << iShift, (iMaxVal + 1 - i) << iShift, 0);
	for (i = 1; i <= iMaxVal; ++ i)
		pColorsOut[iColor ++] = RGB(iMaxVal << iShift, 0, i << iShift);
}

void CreatePowerColorGradient24bit(COLORREF *pColorsOut, int nColorsOut)
{
	int i;
	int iColor = 0;

	_ASSERT(nColorsOut == 6 * 255 + 1);

	for (i = 0; i < 256; ++ i)
		pColorsOut[iColor ++] = RGB(0, 0, i);
	for (i = 1; i < 256; ++ i)
		pColorsOut[iColor ++] = RGB(0, i, 255);
	for (i = 1; i < 256; ++ i)
		pColorsOut[iColor ++] = RGB(0, 255, 256 - i);
	for (i = 1; i < 256; ++ i)
		pColorsOut[iColor ++] = RGB(i, 255, 0);
	for (i = 1; i < 256; ++ i)
		pColorsOut[iColor ++] = RGB(255, 256 - i, 0);
	for (i = 1; i < 256; ++ i)
		pColorsOut[iColor ++] = RGB(255, 0, i);
}


void CreatePowerColorGradient(COLORREF *pColorsOut, int nColorsOut)
{
//	CreateColorGradient(0.0, 1.0, 1.0, 0.0, 1.0, 0.0, pColorsOut, nColorsOut);
//	CreateColorGradient(0.7, -0.8, 0.5, 0.5, 0.7, 0.3, pColorsOut, nColorsOut);
//	CreateColorGradient(0.7, -0.8, 0.5, 1.0, 0.6, 0.8, pColorsOut, nColorsOut);

	double aGradient[] = {
		160.0,  0.0,    0.0,
		160.0, 240.0,  30.0,
		160.0, 240.0,  60.0,
		160.0, 240.0, 102.0,
		 38.0,  58.0, 114.0,
		 41.0, 145.0, 109.0,
		 41.0, 166.0, 158.0,
		 40.0, 240.0, 120.0,
		  8.0, 240.0, 120.0
	};

	int iSegment;

	for (iSegment = 0; iSegment < 8; ++ iSegment) {
		double *pColor = aGradient + iSegment * 3;
		CreateColorGradient(
			pColor[0] / 240.0, pColor[1] / 240.0, pColor[2] / 240.0,
			pColor[3] / 240.0, pColor[4] / 240.0, pColor[5] / 240.0,
			pColorsOut + iSegment * 256, 256);
	}
}

COLORREF HSVtoRGB(double dHue, double dSat, double dVal)
{
	double	dR, dG, dB;
	int		iR, iG, iB;

	while (dHue < 0.0)
		dHue += 360.0;
	while (dHue > 360.0)
		dHue -= 360.0;

	if (dSat < 0.0)
		dSat = 0.0;
	else if (dSat > 1.0)
		dSat = 1.0;
	
	if (dVal < 0.0)
		dVal = 0.0;
	else if (dVal > 1.0)
		dVal = 1.0;

	if (dSat == 0.0)
		dR = dVal,	dG = dVal, dB = dVal;
	else {
		int    iHue = (int)(dHue / 60.0);
		double dF   = dHue / 60.0 - (double)iHue;
		double dP   = dVal * (1.0 - dSat);
		double dQ   = dVal * (1.0 - dF * dSat);
		double dT   = dVal * (1.0 - (1.0 - dF) * dSat);
		switch (iHue) {
		case 0:		dR = dVal,	dG = dT,	dB = dP;	break;
		case 1:		dR = dQ,	dG = dVal,	dB = dP;	break;
		case 2:		dR = dP,	dG = dVal,	dB = dT;	break;
		case 3:		dR = dP,	dG = dQ,	dB = dVal;	break;
		case 4:		dR = dT,	dG = dP,	dB = dVal;	break;
		default:	dR = dVal,	dG = dP,	dB = dQ;	break;
		}
	}

	iR = (int)(dR * 255.0 + 0.5);
	iG = (int)(dG * 255.0 + 0.5);
	iB = (int)(dB * 255.0 + 0.5);

	if (iR < 0)
		iR = 0;
	else if (iR > 255)
		iR = 255;

	if (iG < 0)
		iG = 0;
	else if (iG > 255)
		iG = 255;

	if (iB < 0)
		iB = 0;
	else if (iB > 255)
		iB = 255;

	return RGB(iR, iG, iB);
}

double Hue_2_RGB(double v1, double v2, double vH)
{
   if (vH < 0.0) vH += 1.0;
   if (vH > 1.0) vH -= 1.0;
   if ((6.0 * vH) < 1.0) return v1 + (v2 - v1) * 6.0 * vH;
   if ((2.0 * vH) < 1.0) return v2;
   if ((3.0 * vH) < 2.0) return v1 + (v2 - v1) * ((2.0 / 3.0) - vH) * 6.0;
   return v1;
}

/// HSL values = 0  1
/// RGB results = 0  255
COLORREF HSV_2_RGB(double H, double S, double L)
{
	double	dR, dG, dB;
	int		iR, iG, iB;

	if (S == 0) {
		dR = L;
		dG = L;
		dB = L;
	} else {
		double var_2 = (L < 0.5) ? (L * (1 + S)) : ((L + S) - (S * L));
		double var_1 = 2 * L - var_2;
		dR = Hue_2_RGB(var_1, var_2, H + (1.0 / 3.0));
		dG = Hue_2_RGB(var_1, var_2, H);
		dB = Hue_2_RGB(var_1, var_2, H - (1.0 / 3.0));
	}

	iR = (int)(dR * 255.0 + 0.5);
	iG = (int)(dG * 255.0 + 0.5);
	iB = (int)(dB * 255.0 + 0.5);

	if (iR < 0)
		iR = 0;
	else if (iR > 255)
		iR = 255;

	if (iG < 0)
		iG = 0;
	else if (iG > 255)
		iG = 255;

	if (iB < 0)
		iB = 0;
	else if (iB > 255)
		iB = 255;

	return RGB(iR, iG, iB);
}
