/*
 *    cwtx.c  --  morse code modulator
 *
 *    Copyright (C) 2004
 *      Lawrence Glaister (ve7it@shaw.ca)
 *    This modem borrowed heavily from other gmfsk modems and
 *    also from the unix-cw project. I would like to thank those
 *    authors for enriching my coding experience by providing
 *    and supporting open source.

 *    This file is part of gMFSK.
 *
 *    gMFSK is free software; you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation; either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    gMFSK is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with gMFSK; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

#include "../main/trx.h"
#include "cw.h"
#include "morse.h"
#include "../misc/mixer.h"
#include "../main/sndmodem.h"

/* Define the amplitude envelop for key down events (32 samples long)      */
/* this is 1/2 cycle of a raised cosine                                    */
/* the tables with 32 entries give about 4ms rise and fall times           */
/* when using 8000 samples/sec. This shaping of the cw pulses is           */
/* very necssary to avoid having a very wide and clicky cw signal          */
/* when using the sound card to gen cw. When using the rig key input       */
/* the shaping is done in the rig hardware, but we want to be able to      */
/* pick one cw signal out of a cluster and be able to respond on his freq. */
#define KNUM 32
USHORT kdshape[KNUM] = {
	158,	
	630,
	1411,
	2494,
	3869,
	5522,
	7438,
	9597,
	11980,
	14563,
	17321,
	20228,
	23256,
	26375,
	29556,
	32768,
	35980,
	39161,
	42280,
	45308,
	48215,
	50973,
	53556,
	55938,
	58098,
	60014,
	61667,
	63042,
	64125,
	64906,
	65378,
	65535
};

USHORT kushape[KNUM] = {
	65535,
	65378,
	64906,
	64125,
	63042,
	61667,
	60014,
	58098,
	55938,
	53556,
	50973,
	48215,
	45308,
	42280,
	39161,
	35980,
	32768,
	29556,
	26375,
	23256,
	20228,
	17321,
	14563,
	11980,
	9597,
	7438,
	5522,
	3869,
	2494,
	1411,
	630,
	158
};

/*
=====================================================================
 send_symbol()
 Sends a part of a morse character (one dot duration) of either
 sound at the correct freq or silence. Rise and fall time is controlled
 with a raised cosine shape.
=======================================================================
*/
static void send_symbol(Trx *trx, int symbol)
{
	struct cw *s = (struct cw *) trx->modem;
	int i = 0;
	static int lastkey = 0;
	unsigned iPhaseDelta = (unsigned)((65536.0 * 65536.0) * trx->frequency / (SampleRate << FREQ_FRAC_BITS));

	if (symbol == 1) {
		if (lastkey == 0) {
			/* key going down */
			for (; i < KNUM; i++) {
				trx->outbuf[i] = (mixer_osc_16(s->phaseacc) * kdshape[i]) >> 17;
				s->phaseacc += iPhaseDelta;
			}
		}
		for (; i < s->symbollen; i++) {
			trx->outbuf[i] = mixer_osc_16(s->phaseacc) >> 1;
			s->phaseacc += iPhaseDelta;
		}
	} else {
		/* key going up */
		if (lastkey == 1) {
			for (; i < KNUM; i++) {
				trx->outbuf[i] = (mixer_osc_16(s->phaseacc) * kushape[i]) >> 17;
				s->phaseacc += iPhaseDelta;
			}
		}
		for (; i < s->symbollen; i++) {
			trx->outbuf[i] = 0;
			s->phaseacc += iPhaseDelta;
		}
	}

	sndmodem_write(trx->pModem, trx->outbuf, s->symbollen);
	lastkey = symbol;
}

/*
=====================================================================
send_ch()
sends a morse character and the space afterwards
=======================================================================
*/
static void send_ch(Trx *trx, int symbol)
{
	struct cw *s = (struct cw *) trx->modem;
	int code;

//	fprintf(stderr,"sending %c\n",(char)symbol);

	cw_sync_parameters(s);


	/* handle word space separately (7 dots spacing) */
	/* last char already had 2 dots of inter-character spacing sent with it */
	if ((symbol == ' ') || (symbol == '\n')) {
		send_symbol(trx, 0);
		send_symbol(trx, 0);
		send_symbol(trx, 0);
		send_symbol(trx, 0);
		send_symbol(trx, 0);
		trx_put_echo_char(trx, symbol);
		return;
	}

	/* convert character code to a morse representation */
	if ((symbol <= 255) && (symbol >= 0))
		code = cw_tx_lookup(symbol);
	else
		code = 0x4L;

//	fprintf(stderr,"code 0x%08X\n", code);

	/* loop sending out binary bits of cw character */
	while (code > 1) {
		send_symbol(trx, code & 1);
		code = code >> 1;
	}
	if (symbol != 0)
		trx_put_echo_char(trx, symbol);
}

/*
=====================================================================
 cw_txprocess()
 Read charcters from screen and send them out the sound card.
 This is called repeatedly from a thread during tx.
=======================================================================
*/
int cw_txprocess(Trx *trx)
{
//	struct cw *s = (struct cw *) trx->modem;
	int c;

	if (trx->tune) {
		trx->tune = 0;
		send_symbol(trx, 1);
		return 0;
	}

	c = trx_get_tx_char(trx);

	/* TX buffer empty */
	if (c == -1) {
		/* stop if requested to... */
		if (trx->stopflag)
			return -1;

		send_symbol(trx, 0);
		return 0;
	}

	send_ch(trx, c);

	return 0;
}
