/*****************************************************************************

Programmbeispiel 10: taster_wuerfel

Funktion:
Initialisiert Port B (I/O-Pins auf Ausgang, Ausgnge auf 1)
Initialisiert Port D (I/O-Pins auf Ausgang, Ausgnge auf 1)
In der durch Timer 0 ausgelsten Interruptroutine wird ein Zustandsautomat
aufgerufen, der schrittweise die Muster eines Wrfels ausgibt.
ber eine Zhlvariable wird die Wartezeit bis zum jeweils nchsten Aufruf
des Zustandsautomaten bestimmt.
Der Wrfel wird auf Tasterdruck gestartet und rollt dann langsam aus.

Autor: Kai Ludwig / Version: 1.01

Copyright 2005-2006 Talentraspel Elektronikversand K. Ludwig

Historie:
V1.00 - Erste Version
V1.01 - Anpassung an WinAVR-20060421

*****************************************************************************/

#include <inttypes.h>                // Allgemeine Bibliotheken
#include <avr/io.h>
#include <avr/interrupt.h>

#define sbi(ADDRESS,BIT) ((ADDRESS) |= (1<<(BIT)))
#define cbi(ADDRESS,BIT) ((ADDRESS) &= ~(1<<(BIT)))

#define WUERFEL_1 0x08               // Muster fr 1
#define WUERFEL_2 0x41               // Muster fr 2
#define WUERFEL_3 0x2A               // Muster fr 3
#define WUERFEL_4 0x63               // Muster fr 4
#define WUERFEL_5 0x6B               // Muster fr 5
#define WUERFEL_6 0x77               // Muster fr 6

#define TIMER0_STARTWERT 0x64        // Startwert fr Timer-0
volatile uint8_t zustand;            // Variable fr Zustandsautomat
volatile uint16_t zaehler;           // Zhler fr Zeitsteuerung
volatile uint16_t zeit;              // Variable fr Wrfeltempo

uint8_t taster(void) {               // Tasterabfrage durch kurzes Umschalten
    uint8_t status, ret_val;         // auf Eingang
    
    status=PORTD&0x68;               // Alten Status merken
    DDRD=DDRD&0x97;                  // Bits auf Eingang schalten
    PORTD=PORTD|0x68;

    asm volatile ("nop");
    asm volatile ("nop");
    asm volatile ("nop");
    asm volatile ("nop");
    
    ret_val=PIND&0x68;               // Bits einlesen
    
    PORTD=(PORTD&0x97)|status;       // Alten Status wieder herstellen
    DDRD=DDRD|0x68;                  // Bits wieder auf Ausgang schalten
    
    return ret_val;
}

SIGNAL(SIG_OVERFLOW0) {
    TCNT0 = TIMER0_STARTWERT;        // Timer-0 Startwert setzen
    if (taster()!=0x68) {            // Auf Taster warten
        if (zustand==0) {            // Wenn Wrfel steht, dann
            zustand=1;               // Wrfel starten
            PORTB=0x00;
        }
        zeit=100;                    // Wrfeltempo setzen
        zaehler=0;                   // Wrfelmuster schnell wechseln
    }
    if (zaehler>0)                   // Zhler herunterzhlen
        zaehler--;
    else {
        switch (zustand) {
            case 1: zustand=2;       // Zustand 1
                    PORTD=~WUERFEL_1;
                    zaehler=zeit;
                    break;
            case 2: zustand=3;       // Zustand 2
                    PORTD=~WUERFEL_2;
                    zaehler=zeit;
                    break;
            case 3: zustand=4;       // Zustand 3
                    PORTD=~WUERFEL_3;
                    zaehler=zeit;
                    break;
            case 4: zustand=5;       // Zustand 4
                    PORTD=~WUERFEL_4;
                    zaehler=zeit;
                    break;
            case 5: zustand=6;       // Zustand 5
                    PORTD=~WUERFEL_5;
                    zaehler=zeit;
                    break;
            case 6: zustand=1;       // Zustand 6
                    PORTD=~WUERFEL_6;
                    zaehler=zeit;
                    break;
        }
        if (zustand!=0) {            // Wenn der Wrfel noch luft
            if (zeit<1000)           // Wrfel langsam ausrollen
                zeit=zeit+100;
            else {                   // Wenn Wrfel ausgerollt ist,
                zustand=0;           // dann Wrfel stoppen
                PORTB=0xFF;
            }
        }
    }
}

int main (void) {
    DDRB=0xFF;                       // Port B auf Ausgang
    PORTB=0xFF;                      // Alle Ausgnge auf 1

    DDRD=0xFF;                       // Port D auf Ausgang
    PORTD=0xFF;                      // Alle Ausgnge auf 1

    // Timer-0 Vorteiler auf 64,
    // Startwert setzen und Interrupt einschalten
    TCCR0=(0<<CS02)|(1<<CS01)|(1<<CS00);
    TCNT0=TIMER0_STARTWERT;
    sbi(TIMSK,TOIE0);

    zustand=0;                       // Startzustand
    zaehler=0;                       // Wartezeit fr ersten Aufruf
                                     // des Zustandsautomaten

    sei();                           // Interrupts aktivieren
    
    for (;;) {}                      // Endlosschleife
}
