/*****************************************************************************

Programmbeispiel 09: taster_ampel

Funktion:
Initialisiert Port B (I/O-Pins auf Ausgang, Ausgnge auf 1)
Initialisiert Port D (I/O-Pins auf Ausgang, Ausgnge auf 1)
In der durch Timer 0 ausgelsten Interruptroutine wird ein Zustandsautomat
aufgerufen, der schrittweise die Phasen einer Doppelampel ausgibt.
ber eine Zhlvariable wird die Wartezeit bis zum jeweils nchsten Aufruf
des Zustandsautomaten bestimmt.
Der Zustandsautomat startet mit einer Grnphase fr die erste Ampel und
leitet auf Tasterdruck nach einer kurzen Wartezeit die Grnphase fr die
zweite Ampel ein.

Autor: Kai Ludwig / Version: 1.01

Copyright 2005-2006 Talentraspel Elektronikversand K. Ludwig

Historie:
V1.00 - Erste Version
V1.01 - Anpassung an WinAVR-20060421

*****************************************************************************/

#include <inttypes.h>                // Allgemeine Bibliotheken
#include <avr/io.h>
#include <avr/interrupt.h>

#define sbi(ADDRESS,BIT) ((ADDRESS) |= (1<<(BIT)))
#define cbi(ADDRESS,BIT) ((ADDRESS) &= ~(1<<(BIT)))

#define PHASE_ROT_ROTGELB 0x64       // Muster fr Ampelphasen
#define PHASE_ROT_GRUEN   0x41
#define PHASE_ROT_GELB    0x44
#define PHASE_ROTGELB_ROT 0x70
#define PHASE_GRUEN_ROT   0x22
#define PHASE_GELB_ROT    0x30
#define PHASE_ROT_ROT     0x60

#define TIMER0_STARTWERT 0x64        // Startwert fr Timer-0
volatile uint8_t zustand;            // Variable fr Zustandsautomat
volatile uint16_t zaehler;           // Zhler fr Zeitsteuerung

uint8_t taster(void) {               // Tasterabfrage durch kurzes Umschalten
    uint8_t status, ret_val;         // auf Eingang
    
    status=PORTD&0x68;               // Alten Status merken
    DDRD=DDRD&0x97;                  // Bits auf Eingang schalten
    PORTD=PORTD|0x68;

    asm volatile ("nop");
    asm volatile ("nop");
    asm volatile ("nop");
    asm volatile ("nop");
    
    ret_val=PIND&0x68;               // Bits einlesen
    
    PORTD=(PORTD&0x97)|status;       // Alten Status wieder herstellen
    DDRD=DDRD|0x68;                  // Bits wieder auf Ausgang schalten
    
    return ret_val;
}

SIGNAL(SIG_OVERFLOW0) {
    TCNT0 = TIMER0_STARTWERT;        // Timer-0 Startwert setzen
    if (zaehler>0)                   // Zhler herunterzhlen
        zaehler--;
    else {
        switch (zustand) {
            case 1: zustand=2;            // Zustand 1
                    PORTD=~PHASE_ROT_ROT;
                    zaehler=5000;
                    break;
            case 2: zustand=3;            // Zustand 2
                    PORTD=~PHASE_ROT_ROTGELB;
                    zaehler=1250;
                    break;
            case 3: if (taster()!=0x68) { // Zustand 3
                        zustand=4;        // Auf Taster warten
                        PORTB=0x00;
                        zaehler=2500;
                    }
                    PORTD=~PHASE_ROT_GRUEN;
                    break;
            case 4: zustand=5;            // Zustand 4
                    PORTD=~PHASE_ROT_GELB;
                    zaehler=2500;
                    break;
            case 5: zustand=6;            // Zustand 5
                    PORTD=~PHASE_ROT_ROT;
                    zaehler=5000;
                    break;
            case 6: zustand=7;            // Zustand 6
                    PORTD=~PHASE_ROTGELB_ROT;
                    zaehler=1250;
                    break;
            case 7: zustand=8;            // Zustand 7
                    PORTB=0xFF;
                    PORTD=~PHASE_GRUEN_ROT;
                    zaehler=5000;
                    break;
            case 8: zustand=1;            // Zustand 8
                    PORTD=~PHASE_GELB_ROT;
                    zaehler=2500;
                    break;
        }
    }
}

int main (void) {
    DDRB=0xFF;                       // Port B auf Ausgang
    PORTB=0xFF;                      // Alle Ausgnge auf 1

    DDRD=0xFF;                       // Port D auf Ausgang
    PORTD=0xFF;                      // Alle Ausgnge auf 1

    // Timer-0 Vorteiler auf 64,
    // Startwert setzen und Interrupt einschalten
    TCCR0=(0<<CS02)|(1<<CS01)|(1<<CS00);
    TCNT0=TIMER0_STARTWERT;
    sbi(TIMSK,TOIE0);

    zustand=3;                       // Startzustand
    zaehler=0;                       // Wartezeit fr ersten Aufruf
                                     // des Zustandsautomaten

    sei();                           // Interrupts aktivieren
    
    for (;;) {}                      // Endlosschleife
}
