/*****************************************************************************

Programmbeispiel 06a: parallel_1

Funktion:
Initialisiert Port B (I/O-Pins auf Ausgang, Ausgnge auf 1)
Initialisiert Port D (I/O-Pins auf Ausgang, Ausgnge auf 1)
Gleichzeitiges Blinken der Leuchtdioden und Ausgabe der Phasen
einer Doppelampel

Funktion der Interruptroutine:
In der durch Timer 0 ausgelsten Interruptroutine wird ein Zustandsautomat
aufgerufen, der schrittweise die Phasen einer Doppelampel ausgibt.
ber eine Zhlvariable wird die Wartezeit bis zum jeweils nchsten Aufruf
des Zustandsautomaten bestimmt.

Funktion der Hauptschleife:
In der Hauptschleife werden die Ausgnge abwechselnd auf 0 und 1 gesetzt.
Die Blinkfrequenz wird ber eine Warteschleife in Verbindung mit der
Interruptroutine gesteuert.

Autor: Kai Ludwig / Version: 1.01

Copyright 2005-2006 Talentraspel Elektronikversand K. Ludwig

Historie:
V1.00 - Erste Version
V1.01 - Anpassung an WinAVR-20060421

*****************************************************************************/

#include <inttypes.h>                // Allgemeine Bibliotheken
#include <avr/io.h>
#include <avr/interrupt.h>

#define sbi(ADDRESS,BIT) ((ADDRESS) |= (1<<(BIT)))
#define cbi(ADDRESS,BIT) ((ADDRESS) &= ~(1<<(BIT)))

#define PHASE_ROT_ROTGELB 0x64       // Muster fr Ampelphasen
#define PHASE_ROT_GRUEN   0x41
#define PHASE_ROT_GELB    0x44
#define PHASE_ROTGELB_ROT 0x70
#define PHASE_GRUEN_ROT   0x22
#define PHASE_GELB_ROT    0x30
#define PHASE_ROT_ROT     0x60

#define TIMER0_STARTWERT 0x64        // Startwert fr Timer-0
volatile uint8_t zustand;            // Variable fr Zustandsautomat
volatile uint16_t zaehler_1;         // Zhler 1 fr Zeitsteuerung
volatile uint16_t zaehler_2;         // Zhler 2 fr Zeitsteuerung
volatile uint8_t zaehler_2_fertig;   // Flag fr Ablauf Zhler 2

void warte_ms(uint16_t t) {          // Funktion "Warteschleife" mit Zhler
    cli();                           // ber Interrupt
    zaehler_2=t;                     // Startwert setzen, wegen 16-Bit-Variable
                                     // dabei den Interrupt sperren
    zaehler_2_fertig=0;              // Flag fr Zhlerablauf lschen
    sei();
    while (zaehler_2_fertig==0) {}   // Warten, bis Zhler bei Null ankommt
}

SIGNAL(SIG_OVERFLOW0) {
    TCNT0 = TIMER0_STARTWERT;        // Timer-0 Startwert setzen
    if (zaehler_1>0)                 // Zhler 1 herunterzhlen
        zaehler_1--;
    else {
        switch (zustand) {
            case 1: zustand=2;       // Zustand 1
                    PORTD=~PHASE_ROT_ROT;
                    zaehler_1=5000;
                    break;
            case 2: zustand=3;       // Zustand 2
                    PORTD=~PHASE_ROT_ROTGELB;
                    zaehler_1=1250;
                    break;
            case 3: zustand=4;       // Zustand 3
                    PORTD=~PHASE_ROT_GRUEN;
                    zaehler_1=5000;
                    break;
            case 4: zustand=5;       // Zustand 4
                    PORTD=~PHASE_ROT_GELB;
                    zaehler_1=2500;
                    break;
            case 5: zustand=6;       // Zustand 5
                    PORTD=~PHASE_ROT_ROT;
                    zaehler_1=5000;
                    break;
            case 6: zustand=7;       // Zustand 6
                    PORTD=~PHASE_ROTGELB_ROT;
                    zaehler_1=1250;
                    break;
            case 7: zustand=8;       // Zustand 7
                    PORTD=~PHASE_GRUEN_ROT;
                    zaehler_1=5000;
                    break;
            case 8: zustand=1;       // Zustand 8
                    PORTD=~PHASE_GELB_ROT;
                    zaehler_1=2500;
                    break;
        }
    }
    if (zaehler_2>0)                 // Zhler 2 herunterzhlen
        zaehler_2--;
    else
        zaehler_2_fertig=1;          // Flag fr Zhlerablauf setzen
}

int main (void) {
    DDRB=0xFF;                       // Port B auf Ausgang
    PORTB=0xFF;                      // Alle Ausgnge auf 1

    DDRD=0xFF;                       // Port D auf Ausgang
    PORTD=0xFF;                      // Alle Ausgnge auf 1

    // Timer-0 Vorteiler auf 64,
    // Startwert setzen und Interrupt einschalten
    TCCR0=(0<<CS02)|(1<<CS01)|(1<<CS00);
    TCNT0=TIMER0_STARTWERT;
    sbi(TIMSK,TOIE0);

    zaehler_1=0;                     // Zhlvariable auf Startwert setzen

    zustand=1;                       // Startzustand

    sei();                           // Interrupts aktivieren

    for (;;) {                       // Endlosschleife
        PORTB=~PORTB;                // Ausgnge invertieren
        warte_ms(1000);              // Warten (1000 ms)
    }
}
